// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Step condition routines.

#include "midialf/cond.h"
#include "midialf/dev.h"
#include "midialf/ui.h"

#include "avrlib/string.h"
#include "avrlib/random.h"

namespace midialf {

using namespace avrlib;

const prog_uint8_t random_conditions_percents[kNumRandomConditions] PROGMEM = {
  1, 2, 4, 6, 9, 13, 19, 25, 33, 41, 50, 59, 67, 75, 81, 87, 91, 94, 96, 98, 99,
};

const prog_uint8_t ab_conditions_group_starts[kNumAbConditionGroups] PROGMEM = {
  abFirst,
  abFirst + 2,
  abFirst + 2 + 3,
  abFirst + 2 + 3 + 4,
  abFirst + 2 + 3 + 4 + 5,
  abFirst + 2 + 3 + 4 + 5 + 6,
  abFirst + 2 + 3 + 4 + 5 + 6 + 7,
};

static RunningCallback prev_running_callback;
static StepCallback prev_step_callback;

/* extern */
Cond cond;

/* <static> */
uint16_t Cond::step_count_;
uint16_t Cond::play_count_;
uint8_t Cond::last_result_[kNumConditionTargets];
/* </static> */

/* static */
void Cond::Init() {
  prev_running_callback = dev.set_running_callback(RunningCallback);
  prev_step_callback = dev.set_step_callback(StepCallback);
}

/* static */
uint8_t Cond::CheckCondition(uint8_t condition, uint8_t condition_target) {
  uint8_t result = Evaluate(condition, last_result_[condition_target]);
  if (condition != hasPrevious && condition != noPrevious) {
    last_result_[condition_target] = result;
  }
  return result;
}

/* static */
uint8_t Cond::Evaluate(uint8_t condition, uint8_t last_result) {

  if (condition < abFirst) {
    switch (condition) {
      case hasShift: return ui.IsShifted() ? 1 : 0;
      case noShift: return ui.IsShifted() ? 0 : 1;
      case hasPrevious: return last_result ? 1 : 0;
      case noPrevious: return last_result ? 0 : 1;
      case isFirst: return IsFirst() ? 1 : 0;
      case notFirst:return IsFirst() ? 0 : 1;
    }
    return 0;
  }

  { uint8_t a, b;
    if (get_ab(condition, a, b)) {
      return (play_count_ % b) == (a - 1);
    }
  }

  { uint8_t percent;
    if (get_random(condition, percent)) {
      uint8_t rnd = Random::Get(0, 99);
      return rnd < percent;
    }
  }

  return 0;
}

/* static */
void Cond::get_name(uint8_t condition, char buffer[4]) {

  static const prog_char pchConditions[] PROGMEM = " SHF!SHF PRE!PRE 1ST!1ST";
  if (condition < abFirst) {
    memcpy_P(buffer, &pchConditions[condition << 2], 4);
    return;
  }

  { uint8_t a, b;
    if (get_ab(condition, a, b)) {
      buffer[1] = '0' + a;
      buffer[2] = ':';
      buffer[3] = '0' + b;
      return;
    }
  }

  { uint8_t percent;
    if (get_random(condition, percent)) {
      uint8_t x = 1 + UnsafeItoaLen(percent, 2, &buffer[1]);
      buffer[x] = '%';
      return;
    }
  }

  buffer[0] = '?';
  UnsafeItoa(condition, 3, &buffer[1]);
}

/* static */
uint8_t Cond::get_ab(uint8_t condition, uint8_t& a, uint8_t& b) {
  if (condition < abFirst || condition > abLast)
    return 0;

  for (uint8_t n = 0; n < kNumAbConditionGroups; n++) {
    uint8_t group_start = ResourcesManager::Lookup<uint8_t, uint8_t>(ab_conditions_group_starts, n);
    if (condition < group_start + 2 + n) {
      a = 1 + condition - group_start;
      b = 2 + n;
      return 1;
    }
  }

  return 0;
}

/* static */
uint8_t Cond::get_random(uint8_t condition, uint8_t& percent) {
  if (condition < random1 || condition > random99)
    return 0;

  percent = ResourcesManager::Lookup<uint8_t, uint8_t>(random_conditions_percents, condition - random1);

  return 1;
}

/* static */
uint8_t Cond::IsFirst() {
  switch (dev.seq_link_mode()) {
    case SEQ_LINK_MODE_NONE: return step_count_ < kNumSteps;
    case SEQ_LINK_MODE_16: return step_count_ < kNumSteps * 2;
    case SEQ_LINK_MODE_32: return step_count_ < kNumSteps * 4;
  }

  return 0;
}

/* static */
void Cond::ClearResults() {
  memset(last_result_, 0, sizeof(last_result_));
}

/* static */
void Cond::RunningCallback() {
  step_count_ = -1;
  play_count_ = -1;
  ClearResults();

  if (dev.running()) {
    Random::Seed(milliseconds());
  }

  if (prev_running_callback) {
    prev_running_callback();
  }
}

/* static */
void Cond::StepCallback() {
  ++step_count_;

  if (dev.IsFirstStep()) {
    ++play_count_;
    ClearResults();
  }

  if (prev_step_callback) {
    prev_step_callback();
  }
}

}  // namespace midialf

