// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Device model implementation.

#include "avrlib/serial.h"
#include "avrlib/string.h"
#include "avrlib/random.h"
#include "avrlib/clamp.h"

#include "midi/midi.h"

#include "midialf/midialf.h"
#include "midialf/event_scheduler.h"
#include "midialf/midi_handler.h"
#include "midialf/storage.h"
#include "midialf/scale.h"
#include "midialf/clock.h"
#include "midialf/eemem.h"
#include "midialf/sysex.h"
#include "midialf/leds.h"
#include "midialf/dev.h"
#include "midialf/lfo.h"
#include "midialf/ui.h"

namespace midialf {

using namespace avrlib;

/* extern */
Dev dev;

/* <static> */
uint8_t Dev::slot_;
uint8_t Dev::name_[kNameLength];
uint8_t Dev::channel_ = kDefChannel;
uint8_t Dev::tempo_ = kDefTempo;
uint8_t Dev::step_length_ = kDefStepLength;
int16_t Dev::swing_ = kDefSwing;
uint8_t Dev::seq_root_note_ = kDefSeqRootNote;
uint8_t Dev::seq_link_mode_ = kDefSeqLinkMode;
uint8_t Dev::seq_switch_mode_ = kDefSeqSwitchMode;
uint8_t Dev::seq_transpose_mode_ = kDefSeqTransposeMode;
uint8_t Dev::seq_direction_ = kDefSeqDirection;
uint8_t Dev::steps_forward_ = kDefStepCount;
uint8_t Dev::steps_backward_ = kDefStepCount;
uint8_t Dev::steps_replay_ = kDefStepCount;
uint8_t Dev::steps_interval_ = kDefStepCount;
uint8_t Dev::steps_repeat_ = kDefStepCount;
uint8_t Dev::steps_skip_ = kDefStepCount;
uint8_t Dev::cc1_number_ = kDefCC1Number;
uint8_t Dev::cc2_number_ = kDefCC2Number;
uint8_t Dev::prog_change_mode_ = kDefProgChangeMode;
uint8_t Dev::ctrl_change_mode_ = kDefCtrlChangeMode;
uint8_t Dev::clock_mode_ = kDefClockMode;
uint8_t Dev::bank_select_msb_ = kDefBankSelectMsb;
uint8_t Dev::bank_select_lsb_ = kDefBankSelectLsb;
uint8_t Dev::program_change_ = kDefProgramChange;
uint8_t Dev::song_step_;
uint8_t Dev::song_step_play_count_;
uint16_t Dev::flags_ = kDefFlags;
#ifdef ENABLE_CV_EXT
uint8_t Dev::cv_mode_[4];
uint8_t Dev::gate_mode_[4];
uint8_t Dev::strobe_width_;
#endif

uint8_t Dev::tick_;
uint8_t Dev::running_;
uint8_t Dev::recording_;
uint8_t Dev::clock_prescaler_;

uint8_t Dev::view_seq_;
uint8_t Dev::play_seq_;
uint8_t Dev::step_;
uint8_t Dev::step_hold_;
uint8_t Dev::transpose_note_;
uint8_t Dev::pendulum_backward_;
uint8_t Dev::steps_forward_counter_;
uint8_t Dev::steps_replay_counter_;
uint8_t Dev::steps_interval_counter_;
uint8_t Dev::steps_repeat_counter_;
uint8_t Dev::steps_skip_counter_;
uint8_t Dev::saved_replay_step_;
uint8_t Dev::playing_steps_count_;
uint8_t Dev::request_set_seq_;
uint8_t Dev::next_step_;
uint8_t Dev::next_seq_;
uint8_t Dev::manual_step_selected_;
uint8_t Dev::last_recorded_step_;
uint8_t Dev::last_recorded_seq_;
uint8_t Dev::last_legato_note_ = noNote;

Seq Dev::seq_[kNumSeqs];

Song Dev::song_;

NoteStack<16> Dev::note_stack_;
NoteStack<64> Dev::sent_notes_stack_;

NoteCallback Dev::note_callback_;
ControlChangeCallback Dev::control_change_callback_;
RunningCallback Dev::running_callback_;
TickCallback Dev::tick_callback_;
StepCallback Dev::step_callback_;
/* </static> */

///////////////////////////////////////////////////////////////////////////////
// Device object handling

/* static */
void Dev::Init() {
  UpdateClock();
  clock.Start();
  cond.Init();
  lfo.Init();

  UpdateClockPrescaler();

  LoadSlot(eemem.slot());
}

///////////////////////////////////////////////////////////////////////////////
// MIDI event handlers

/* static */
void Dev::OnNoteOn(uint8_t channel, uint8_t note, uint8_t velocity) {
  if (note_callback_ && note_callback_(channel, note, velocity))
    return;

  if (channel != channel_)
    return;

  if (recording_ || !running_ || seq_transpose_mode_ == SEQ_TRANSPOSE_MODE_NONE) {
    Send3(0x90 | channel, note, velocity);
    note_stack_.NoteOn(note, velocity, channel);
    sent_notes_stack_.NoteOn(note, velocity, channel);
  } else {
    transpose_note_ = note;
  }

  if (recording_) {
    RecordStep(note, velocity);
  }
}

/* static */
void Dev::OnNoteOff(uint8_t channel, uint8_t note, uint8_t velocity) {
  if (note_callback_ && note_callback_(channel, note, 0))
    return;

  if (channel != channel_)
    return;

  if (recording_ || !running_ || seq_transpose_mode_ == SEQ_TRANSPOSE_MODE_NONE) {
    uint8_t most_recent_note = note_stack_.most_recent_note().note;
    note_stack_.NoteOff(note, channel);
    sent_notes_stack_.NoteOff(note, channel);
    Send3(0x80 | channel, note, velocity);
    if (recording_ && note_stack_.size() > 0) {
      // Most recent pressed key has been released, however some keys are
      // still pressed, so we need to record the most recent of them.
      if (most_recent_note == note) {
        RecordStep(
          note_stack_.most_recent_note().note,
          note_stack_.most_recent_note().velocity);
      }
    }
  }
}

/* static */
void Dev::OnAftertouch(uint8_t channel, uint8_t note, uint8_t velocity) {
  if (channel != channel_)
    return;

#ifdef ENABLE_CV_EXT
    cv.SendATch(velocity);
#endif
}

/* static */
void Dev::OnAftertouch(uint8_t channel, uint8_t velocity) {
  if (channel != channel_)
    return;

#ifdef ENABLE_CV_EXT
    cv.SendATch(velocity);
#endif
}

/* static */
void Dev::OnControlChange(uint8_t channel, uint8_t controller, uint8_t value) {
  if (control_change_callback_ && control_change_callback_(channel, controller, value))
    return;

  if (channel != channel_)
    return;

  if (HandleCC(channel, controller, value))
    return;

  if (controller == cc1_number_) {
#ifdef ENABLE_CV_EXT
    cv.SendCC1(value);
#endif
    if (recording_) {
      set_play_seq_step_cc1(step_, value);
      set_play_seq_step_send_cc1(step_, 1);
      ui.request_redraw();
    }
  } else
  if (controller == cc2_number_) {
#ifdef ENABLE_CV_EXT
    cv.SendCC2(value);
#endif
    if (recording_) {
      set_play_seq_step_cc2(step_, value);
      set_play_seq_step_send_cc2(step_, 1);
      ui.request_redraw();
    }
  }
}

/* static */
void Dev::OnProgramChange(uint8_t channel, uint8_t program) {
  if (channel != channel_)
    return;

  if (prog_change_mode_ & PROGRAM_CHANGE_RECV && program < storage.num_slots()) {
    LoadSlot(program);
  }
}

/* static */
void Dev::OnPitchBend(uint8_t channel, uint16_t pitch_bend) {
  if (channel != channel_)
    return;

#ifdef ENABLE_CV_EXT
  cv.SendPBnd(pitch_bend);
#endif
}

/* static */
void Dev::OnSysExByte(uint8_t sysex_byte) {
  sysex.Receive(sysex_byte);
  sysex.Forward(sysex_byte);
}

/* static */
void Dev::OnClock() {
  if (ExternalClock() && running_) {
    Tick();
  }
}

/* static */
void Dev::OnStart() {
  if (ExternalClock()) {
    Start();
  }
}

/* static */
void Dev::OnContinue() {
  if (ExternalClock()) {
    running_ = 1;
  }
}

/* static */
void Dev::OnStop() {
  if (ExternalClock()) {
    Stop();
  }
}

/* static */
uint8_t Dev::CheckChannel(uint8_t channel) {
  return 1;
}

/* static */
void Dev::OnRawByte(uint8_t byte) {
}

/* static */
void Dev::OnRawMidiData(
  uint8_t status,
  uint8_t* data,
  uint8_t data_size,
  uint8_t accepted_channel) {

  // Filter MIDI Clock related events
  switch (status) {
    case 0xf2: // SPP
    case 0xf8: // Clock
    case 0xfa: // Start
    case 0xfb: // Continue
    case 0xfc: // Stop
      // Don't forward clock related events unless configured for external clock and
      // clock output is enabled.
      if (!(ExternalClock() && SendOutputClock()))
        return;
      // Don't forward clock events unless configured appropriately
      if (status == 0xf8 && !(ContiguousClock() || running_))
        return;
  }

  // Forward everything except note events on for the selected channel.
  if (status != (0x80 | channel_) &&
      status != (0x90 | channel_)) {
    Send(status, data, data_size);
  }
}

/* static */
void Dev::OnInternalClockTick() {
  if (InternalClock()) {
    if (SendOutputClock()) {
      if (running_ || ContiguousClock()) {
        SendNow(0xf8);
      }
    }
    if (running_) {
      Tick();
    } else
      SendScheduledNotes();
  } else {
    // Make all notes get their note offs served
    if (!running_) {
      SendScheduledNotes();
    }
  }
}

///////////////////////////////////////////////////////////////////////////////
// Device actions

/* static */
void Dev::InitDev() {
  eemem.Reset();
  InitSlots();
  LoadSlot(0);
#ifdef ENABLE_CV_EXT
  cv.InitTune();
#endif
}

/* static */
void Dev::InitSlots() {
  Slot slot; slot.Init();
  for (uint8_t n = 0; n < kNumSlots; n++) {
    slot.Write(n);
  }
}

/* static */
void Dev::InitSlot() {
  Slot slot; slot.Init();
  slot.Write(slot_);
  LoadSlot(slot_);
}

/* static */
void Dev::LoadSlot(uint8_t slot_numb) {
  set_slot(slot_numb);

  Slot slot; slot.Read(slot_numb);

  FromSlot(slot);

  SendSeqProgramChange();

  SetPlaySeq(0);
}

/* static */
void Dev::SaveSlot(uint8_t slot_numb) {
  set_slot(slot_numb);

  Slot slot; ToSlot(slot);

  slot.Write(slot_numb);
}

/* static */
void Dev::FromSlot(const Slot& slot) {
  set_name(slot.name_);
  set_channel(slot.channel_);
  set_tempo(slot.tempo_);
  set_step_length(slot.step_length_);
  set_swing(slot.swing_);
  set_seq_root_note(slot.seq_root_note_);
  set_seq_link_mode(slot.seq_link_mode_);
  set_seq_switch_mode(slot.seq_switch_mode_);
  set_seq_transpose_mode(slot.seq_transpose_mode_);
  set_seq_direction(slot.seq_direction_);
  set_steps_forward(slot.steps_forward_);
  set_steps_backward(slot.steps_backward_);
  set_steps_replay(slot.steps_replay_);
  set_steps_interval(slot.steps_interval_);
  set_steps_repeat(slot.steps_repeat_);
  set_steps_skip(slot.steps_skip_);
  set_cc1_number(slot.cc1_number_);
  set_cc2_number(slot.cc2_number_);
  set_prog_change_mode(slot.prog_change_mode_);
  set_ctrl_change_mode(slot.ctrl_change_mode_);
  set_clock_mode(slot.clock_mode_);
  set_bank_select_msb(slot.bank_select_msb_);
  set_bank_select_lsb(slot.bank_select_lsb_);
  set_program_change(slot.program_change_);
  set_flags(slot.flags_);
#ifdef ENABLE_CV_EXT
  memcpy(cv_mode_, slot.cv_mode_, sizeof(cv_mode_));
  memcpy(gate_mode_, slot.gate_mode_, sizeof(gate_mode_));
  set_strobe_width(slot.strobe_width_);
#endif

  memcpy(seq_, slot.seq_, sizeof(seq_));

  song_ = slot.song_;

  for (uint8_t n = 0; n < kNumLfos; n++) {
    lfo.set_settings(n, slot.lfo_[n]);
  }

  lfo.set_resolution(slot.lfo_resolution_);
}

/* static */
void Dev::ToSlot(Slot& slot) {
  memcpy(slot.name_, name_, kNameLength);
  slot.channel_ = channel();
  slot.tempo_ = tempo();
  slot.step_length_ = step_length();
  slot.swing_ = swing();
  slot.seq_root_note_ = seq_root_note();
  slot.seq_link_mode_ = seq_link_mode();
  slot.seq_switch_mode_ = seq_switch_mode();
  slot.seq_transpose_mode_ = seq_transpose_mode();
  slot.seq_direction_ = seq_direction();
  slot.steps_forward_ = steps_forward();
  slot.steps_backward_ = steps_backward();
  slot.steps_replay_ = steps_replay();
  slot.steps_interval_ = steps_interval();
  slot.steps_repeat_ = steps_repeat();
  slot.steps_skip_ = steps_skip();
  slot.cc1_number_ = cc1_number();
  slot.cc2_number_ = cc2_number();
  slot.prog_change_mode_ = prog_change_mode();
  slot.ctrl_change_mode_ = ctrl_change_mode();
  slot.clock_mode_ = clock_mode();
  slot.bank_select_msb_ = bank_select_msb();
  slot.bank_select_lsb_ = bank_select_lsb();
  slot.program_change_ = program_change();
  slot.flags_ = flags();
#ifdef ENABLE_CV_EXT
  memcpy(slot.cv_mode_, slot.cv_mode_, sizeof(cv_mode_));
  memcpy(slot.gate_mode_, gate_mode_, sizeof(gate_mode_));
  slot.strobe_width_ = strobe_width();
#endif

  memcpy(slot.seq_, seq_, sizeof(slot.seq_));

  slot.song_ = song_;

  for (uint8_t n = 0; n < kNumLfos; n++) {
    lfo.get_settings(n, slot.lfo_[n]);
  }

  slot.lfo_resolution_ = lfo.resolution();
}

/* static */
void Dev::SetPlaySeq(uint8_t seq) {
  if (play_seq_ == seq)
    return;

  if (running_ && seq_switch_mode_ == SEQ_SWITCH_MODE_ONSEQEND) {
    request_set_seq_ = 1 + seq;
  } else
    set_play_seq(seq);
}

///////////////////////////////////////////////////////////////////////////////
// Core sequencer routines

/* static */
void Dev::ToggleRun() {
  if (!running_) {
    Start();
  } else {
    Stop();
  }
}

/* static */
void Dev::Start() {
  if (running_)
    return;

  if (InternalClock()) {
    if (SendOutputClock()) SendNow(0xfa);
  }

  UpdateClockPrescaler();

#ifdef ENABLE_CV_EXT
  port.SendStart(1);
#endif

  saved_replay_step_ = 0;

  // Set step to start with
  if (step_hold_) {
    ;
  } else
  if (AreAllStepsSkipped()) {
    step_ = 0;
  } else {
    // Set initial step one before the start step to ensure consistent led behavior
    switch (seq_direction_) {

      case SEQ_DIRECTION_FORWARD: {
          uint8_t first_step = (kNumSteps << seq_link_mode_) - 1;
          switch (seq_link_mode_) {
            case SEQ_LINK_MODE_16: play_seq_ = (play_seq_ & 2) + (first_step >> 3); break;
            case SEQ_LINK_MODE_32: play_seq_ = first_step >> 3; break;
          }
          step_ = first_step & (kNumSteps - 1);
        }
        break;

      case SEQ_DIRECTION_BACKWARD:
        step_ = 0;
        break;

      case SEQ_DIRECTION_PENDULUM:
        step_ = 1; pendulum_backward_ = 1;
        switch (seq_link_mode_) {
          case SEQ_LINK_MODE_16: play_seq_&= 2; break;
          case SEQ_LINK_MODE_32: play_seq_ = 0; break;
        }
        break;

      case SEQ_DIRECTION_RANDOM:
        Random::Seed((uint16_t)milliseconds());
        break;
    }

    AdvanceStep();
  }

  // Reset progression controls
  steps_forward_counter_ = 0;
  steps_replay_counter_ = 0;
  steps_interval_counter_ = 0;
  steps_repeat_counter_ = 0;
  steps_skip_counter_ = 0;

  // Reset running controls
  tick_ = clock_prescaler_ - 1;
  last_legato_note_ = noNote;
  transpose_note_ = seq_root_note_;
  request_set_seq_ = 0;
  running_ = -1;

  // Reset song controls
  song_step_ = 0;
  song_step_play_count_ = 0;
  if (song_mode_active()) {
    transpose_note_ = song_step_root_note(0);
  }

  lfo.Start();

  if (running_callback_) {
    running_callback_();
  }
}

/* static */
void Dev::Stop() {
  if (!running_)
    return;

#ifdef ENABLE_CV_EXT
  port.SendStart(0);
#endif

  FlushQueue();

  if (InternalClock()) {
    if (SendOutputClock()) SendNow(0xfc);
  }

  running_ = 0;
  request_set_seq_ = 0;
  last_legato_note_ = noNote;

  lfo.Stop();

  if (running_callback_) {
    running_callback_();
  }
}

/* static */
void Dev::Tick() {
  ++tick_;

#ifdef ENABLE_CV_EXT
  port.SendClock();
#endif

  SendScheduledNotes();

  lfo.Tick();

  if (tick_callback_) {
    tick_callback_();
  }

  if (tick_ < clock_prescaler_)
    return;

  tick_ = 0;

  if (running_ == 1) {
    if (!step_hold_) {
      AdvanceStep();
    }
  } else
    running_ = 1;

  if (step_callback_) {
    step_callback_();
  }

  // Handle sequence select request
  if (step_ == 0) {
    switch (seq_link_mode_) {
      case SEQ_LINK_MODE_16:
        if ((play_seq_ & 1) == 0)
          goto OnSeq;
        break;
      case SEQ_LINK_MODE_32:
        if (play_seq_)
          break;
        // fall through
      default:
OnSeq:  if (request_set_seq_) {
          set_play_seq(request_set_seq_ - 1);
          request_set_seq_ = 0;
          ui.request_redraw();
        }
#ifdef ENABLE_CV_EXT
        port.SendSeq();
#endif
        break;
    }
  }

  // Send step unless condition prevents it
  if (CheckNoteCondition(play_seq_, step_)) {
    SendStep(play_seq_, step_);
  }
}

/* static */
void Dev::AdvanceStep() {
  manual_step_selected_ = 0;

  // If all steps are skipped, do nothing
  playing_steps_count_ = NonSkippedStepsCount();
  if (playing_steps_count_ == 0)
    return;

  // Setup running position
  next_step_ = step_;
  next_seq_ = play_seq_;

  // Make it a virtual step
  switch (seq_link_mode_) {
    case SEQ_LINK_MODE_16: if (play_seq_ & 1) next_step_+= kNumSteps; break;
    case SEQ_LINK_MODE_32: if (play_seq_) next_step_+= play_seq_ * kNumSteps; break;
  }

  // Step progression logic based on MIDIbox SEQ V4 code by Thorsten Klose (tk@midibox.org)
  uint8_t advance = 1;
  uint8_t save_step = 0;

  // Check if progression options are not applicable
  if (playing_steps_count_ > 1 && seq_direction_ != SEQ_DIRECTION_RANDOM) {

    if (steps_forward_ && ++steps_forward_counter_ >= steps_forward_) {
      steps_forward_counter_ = 0;
      if (steps_backward_) {
        for (uint8_t i = 0; i < steps_backward_; ++i) {
          AdvanceStep(GetRunningDirection(1)); // reverse
        }
      }
      if (++steps_replay_counter_ > steps_replay_) {
        steps_replay_counter_ = 0;
        save_step = 1;
      } else {
        next_step_ = saved_replay_step_;
        advance = 0;
      }
    }

    if (steps_interval_ && !steps_repeat_counter_ && !steps_skip_counter_) {
      if (++steps_interval_counter_ >= steps_interval_) {
        steps_interval_counter_ = 0;
        steps_repeat_counter_ = steps_repeat_;
        steps_skip_counter_ = steps_skip_;
      }
    }

    if (steps_repeat_counter_) {
      --steps_repeat_counter_;
      advance = 0;
    } else {
      while(steps_skip_counter_) {
        AdvanceStep(GetRunningDirection(0));  // forward
        --steps_skip_counter_;
      }
    }
  }

  // Advance to the next step if necessary
  if (advance) {
    AdvanceStep(seq_direction_);
  }

  // Select next sequence if song mode is active
  if (song_mode_active() && running_ && !recording_ && next_step_ == 0) {
    if (++song_step_play_count_ >= song_step_play_count(song_step_)) {
      song_step_play_count_ = 0;
      if (song_.SelectNextStep(song_step_)) {
        next_seq_ = song_step_seq(song_step_);
        transpose_note_ = song_step_root_note(song_step_);
      }
    }
  }

  // Advance to the next sequence if any
  if (next_seq_ != play_seq_) {
    set_play_seq(next_seq_);
    ui.request_redraw();
  }

  // Set the actual step to play
  set_step(next_step_ & (kNumSteps - 1));

  // Save step if requested
  if (save_step) {
    saved_replay_step_ = next_step_;
  }

  // Sync lfos
  lfo.OnStep();
}

/* static */
void Dev::AdvanceStep(uint8_t direction) {
  uint8_t current_step = next_step_;

  // Find out number of virtual steps
  uint8_t num_steps = kNumSteps << seq_link_mode_;

  // Advance to the next virtual step in a loop, skipping
  // the skipped steps
  for (;;) {
    switch (direction) {
      case SEQ_DIRECTION_FORWARD:
        if (++next_step_ >= num_steps) {
          next_step_ = 0;
        }
        break;
      case SEQ_DIRECTION_BACKWARD:
        if (next_step_-- == 0) {
          next_step_ = num_steps - 1;
        }
        break;
      case SEQ_DIRECTION_PENDULUM:
        if (pendulum_backward_) {
          if (next_step_-- == 0) {
            pendulum_backward_ = 0;
            next_step_ = 1;
          }
        } else {
          if (++next_step_ >= num_steps) {
            pendulum_backward_ = 1;
            next_step_ = num_steps - 2;
          }
        }
        break;
      case SEQ_DIRECTION_RANDOM:
        do {
          next_step_ = Random::GetByte() & (num_steps - 1);
        } while (next_step_ == current_step && playing_steps_count_ > 1);
        break;
    }

    // Figure out the next sequence
    switch (seq_link_mode_) {
      case SEQ_LINK_MODE_16: next_seq_ = (next_seq_ & 2) + (next_step_ >> 3); break;
      case SEQ_LINK_MODE_32: next_seq_ = next_step_ >> 3; break;
    }

    // Check if next step is not skipped, keep trying if it is
    if (!seq_step_skip(next_seq_, next_step_ & (kNumSteps - 1))) {
      // Check if we've got the same step and keep trying only if we have more than one step to
      // ensure that we activate the next not skipped step
      if (next_step_ == current_step && playing_steps_count_ > 1)
        continue;
      break;
    }
  }
}

/* static */
void Dev::SendStep(uint8_t seq, uint8_t step) {
  SendStepCC(seq, step);

  if (seq_step_mute(seq, step) && !manual_step_selected_)
    return;

  if (!transpose_note_) transpose_note_ = seq_root_note_;

  // Calculate note to send
  uint8_t note = Clamp7F(static_cast<int16_t>(seq_step_note(seq, step))
    + transpose_note_ - seq_root_note_);

  // Randomize note to send

  uint8_t randomize_note = seq_step_randomize_note(seq, step);
  if (randomize_note) {
    note = RandomizeNote(note, randomize_note, seq_step_randomize_note_scale(seq, step));
  }

  // Check if the note on has been sent for this note and if so, send the note off now and
  // also remove it from the scheduler queue
  if (sent_notes_stack_.NoteOff(note, channel_)) {
    event_scheduler.Remove(note, 0, channel_);
    Send3(0x80 | channel_, note, 0);
  }

  uint8_t velocity = Randomize7F(seq_step_velocity(seq, step), seq_step_randomize_velocity(seq, step));
  if (!velocity) velocity = 1;

  // Send the note
  Send3(0x90 | channel_, note, velocity);
  sent_notes_stack_.NoteOn(note, velocity, channel_);

  // Release previous legato note if any
  if (last_legato_note_ != noNote) {
    if (sent_notes_stack_.NoteOff(last_legato_note_, channel_)) {
      Send3(0x80 | channel_, last_legato_note_, 0);
    }
    last_legato_note_ = noNote;
  }

  // Handle retrigger options
  if (seq_step_has_retrigger(seq, step) && CheckRetriggerCondition(seq, step)) {
    SendStepRetrigger(seq, step, note, velocity);
  } else {
    // Schedule note off unless this is a legato note
    if (seq_step_legato(seq, step) && !(step_hold_ || manual_step_selected_)) {
      last_legato_note_ = note;
    } else {
      uint8_t duration = Duration::GetMidiClockTicks(seq_step_gate(seq, step));
      SendLater(note, 0, duration - 1);
    }
  }
}

/* static */
void Dev::SendStepRetrigger(uint8_t seq, uint8_t step, uint8_t note, uint8_t velocity) {
  uint8_t retrigger_count = seq_step_retrigger(seq, step);
  uint8_t step_ticks = Duration::GetMidiClockTicks(step_length());

  uint8_t retrigger_ticks = step_ticks / retrigger_count;
  if (retrigger_ticks == 0) retrigger_ticks = 1;

  uint8_t duration_ticks = retrigger_ticks / 2;
  if (duration_ticks == 0) duration_ticks = 1;

  uint8_t retrigger_when = 0;
  for (uint8_t n = 1; n < retrigger_count; n++) {
    retrigger_when = n * retrigger_ticks;
    SendLater(note, 0, retrigger_when - duration_ticks);

    int8_t transposition =
      static_cast<int16_t>(seq_step_retrigger_transposition(seq, step))
      - kDefStepRetriggerTransposition;

    if (transposition) {
      uint8_t scale = seq_step_retrigger_transposition_scale(seq, step);
      if (scale > 0) {
        // Scale transposition
        uint8_t up = 1;
        if (transposition < 0) {
          transposition = - transposition;
          up = 0;
        }
        for (uint8_t n = 0; n < transposition; n++) {
          note = Scale::GetNextScaledNote(scale - 1, note, up);
        }
      } else {
        // Chromatic transposition
        note = Clamp7F(static_cast<int16_t>(note) + transposition);
     }
    }

    velocity = Clamp(static_cast<int16_t>(velocity)
      + static_cast<int16_t>(seq_step_retrigger_velocity(seq, step))
      - kDefStepRetriggerVelocity,
      1, 0x7F);

    SendLater(note, velocity, retrigger_when - 1);
  }

  SendLater(note, 0, retrigger_when + retrigger_ticks - duration_ticks);
}

/* static */
void Dev::SendStepCC(uint8_t seq, uint8_t step) {
  if (seq_step_send_cc1(seq, step) && CheckCC1Condition(seq, step)) {
    Send3(0xb0 | channel_, cc1_number(),
      Randomize7F(seq_step_cc1(seq, step), seq_step_randomize_cc1(seq, step)));
  }
  if (seq_step_send_cc2(seq, step) && CheckCC2Condition(seq, step)) {
    Send3(0xb0 | channel_, cc2_number(),
      Randomize7F(seq_step_cc2(seq, step), seq_step_randomize_cc2(seq, step)));
  }
}

/* static */
uint8_t Dev::AreAllStepsSkipped() {
  switch (seq_link_mode_) {
    case SEQ_LINK_MODE_NONE:
      return all_steps_skipped();

    case SEQ_LINK_MODE_16:
      return all_steps_skipped(play_seq_ & 2)
        && all_steps_skipped((play_seq_ & 2) + 1);

    case SEQ_LINK_MODE_32:
      return all_steps_skipped(0)
        && all_steps_skipped(1)
        && all_steps_skipped(2)
        && all_steps_skipped(3);
  }
}

/* static */
uint8_t Dev::NonSkippedStepsCount() {
  switch (seq_link_mode_) {
    case SEQ_LINK_MODE_NONE:
      return non_skipped_steps_count();

    case SEQ_LINK_MODE_16:
      return non_skipped_steps_count(play_seq_ & 2)
        + non_skipped_steps_count((play_seq_ & 2) + 1);

    case SEQ_LINK_MODE_32:
      return non_skipped_steps_count(0)
        + non_skipped_steps_count(1)
        + non_skipped_steps_count(2)
        + non_skipped_steps_count(3);
  }
}

/* static */
uint8_t Dev::GetRunningDirection(uint8_t reversed) {
  if (seq_direction_ == SEQ_DIRECTION_BACKWARD
    || (seq_direction_ == SEQ_DIRECTION_PENDULUM && pendulum_backward_)) {
    reversed = !reversed;
  }
  return reversed ? SEQ_DIRECTION_BACKWARD : SEQ_DIRECTION_FORWARD;
}

/* static */
uint8_t Dev::CheckNoteCondition(uint8_t seq, uint8_t step) {
  if (!running_ || !seq_step_has_condition(seq, step)) return 1;
  return cond.CheckCondition(seq_step_condition(seq, step), noteCondition);
}

/* static */
uint8_t Dev::CheckCC1Condition(uint8_t seq, uint8_t step) {
  if (!running_ || !seq_step_has_cc1_condition(seq, step)) return 1;
  return cond.CheckCondition(seq_step_cc1_condition(seq, step), cc1Condition);
}

/* static */
uint8_t Dev::CheckCC2Condition(uint8_t seq, uint8_t step) {
  if (!running_ || !seq_step_has_cc2_condition(seq, step)) return 1;
  return cond.CheckCondition(seq_step_cc2_condition(seq, step), cc2Condition);
}

/* static */
uint8_t Dev::CheckRetriggerCondition(uint8_t seq, uint8_t step) {
  if (!running_ || !seq_step_has_retrigger_condition(seq, step)) return 1;
  return cond.CheckCondition(seq_step_retrigger_condition(seq, step), retriggerCondition);
}

/* static */
uint8_t Dev::IsFirstStep(uint8_t step) {
  // Make it a virtual step
  switch (seq_link_mode_) {
    case SEQ_LINK_MODE_16: if (play_seq_ & 1) step+= kNumSteps; break;
    case SEQ_LINK_MODE_32: if (play_seq_) step+= play_seq_ * kNumSteps; break;
  }

  if (seq_direction_ == SEQ_DIRECTION_BACKWARD) {
    switch (seq_link_mode_) {
      case SEQ_LINK_MODE_NONE: return step == kNumSteps - 1;
      case SEQ_LINK_MODE_16: return step == 2 * kNumSteps - 1;
      case SEQ_LINK_MODE_32: return step == 4 * kNumSteps - 1;
    }
  }

  return step == 0;
}

///////////////////////////////////////////////////////////////////////////////
// Recording mode routines

/* static */
void Dev::StopRecording() {
  if (recording_) {
    recording_ = 0;
  }
}

/* static */
void Dev::StartRecording() {
  if (!recording_) {
    step_ = 0;
    seq_direction_ = SEQ_DIRECTION_FORWARD;
    transpose_note_ = seq_root_note_;
    last_recorded_step_ = noStep;
    recording_ = 1;
  }
}

/* static */
void Dev::ToggleRecording() {
  if (!recording_) {
    StartRecording();
  } else
    StopRecording();
}

/* static */
void Dev::RecordStep(uint8_t note, uint8_t velocity) {
  if (last_recorded_step_ != noStep) {
    uint8_t legato = note_stack_.size() > 1;
    set_seq_step_legato(last_recorded_seq_, last_recorded_step_, legato);
  }

  set_play_seq_step_note(step_, note);
  set_play_seq_step_velocity(step_, velocity);

  last_recorded_step_ = step_;
  last_recorded_seq_ = play_seq_;

  if (!running_) {
    AdvanceStep();
  }

  ui.request_redraw();
}

///////////////////////////////////////////////////////////////////////////////
// Miscellaneous routines

/* static */
void Dev::UpdateSlot() {
  eemem.set_slot(slot_);
}

/* static */
void Dev::UpdateClock() {
  clock.Update(tempo_, swing_);
}

/* static */
void Dev::UpdateClockPrescaler() {
  clock_prescaler_ = ResourcesManager::Lookup<uint8_t, uint8_t>(midi_clock_ticks_per_note, step_length_);
  clock.set_tick_count_limit(clock_prescaler_);
  lfo.UpdatePrescaler();
}

///////////////////////////////////////////////////////////////////////////////
// MIDI output helpers

/* static */
void Dev::SendNow(uint8_t byte) {
  midi_out.Send(byte);
}

/* static */
void Dev::Send2(uint8_t a, uint8_t b) {
  FlushOutputBuffer(2);
  MidiHandler::OutputBuffer::Write(a);
  MidiHandler::OutputBuffer::Write(b);
}

/* static */
void Dev::Send3(uint8_t a, uint8_t b, uint8_t c) {

#ifdef ENABLE_CV_EXT
  if ((a & 0x0f) == channel_) {
    switch (a & 0xf0) {
      case 0x80:
        if (!running_ || recording_) {
          if (note_stack_.size() == 0) {
            port.SendGate(0);
          }
        } else {
          // This check assumes we're only scheduling note offs
          if (event_scheduler.size() <= 1) {
            port.SendGate(0);
          }
        }
        break;
      case 0x90:
        cv.SendNote(b);
        cv.SendVelo(c);
        if (c) {
          port.SendGate(1);
          port.SendStrobe();
        }
        break;
      case 0xb0:
        if (b == cc1_number()) {
          cv.SendCC1(c);
        } else
        if (b == cc2_number()) {
          cv.SendCC2(c);
        }
        break;
    }
  }
#endif

  FlushOutputBuffer(3);
  MidiHandler::OutputBuffer::Write(a);
  MidiHandler::OutputBuffer::Write(b);
  MidiHandler::OutputBuffer::Write(c);
}

/* static */
void Dev::Send(uint8_t status, uint8_t* data, uint8_t size) {
  FlushOutputBuffer(1 + size);
  MidiHandler::OutputBuffer::Write(status);
  if (size) {
    MidiHandler::OutputBuffer::Write(*data++);
    --size;
  }
  if (size) {
    MidiHandler::OutputBuffer::Write(*data++);
    --size;
  }
}

/* static */
void Dev::FlushOutputBuffer(uint8_t requested_size) {
  while (MidiHandler::OutputBuffer::writable() < requested_size) {
    uint8_t byte = MidiHandler::OutputBuffer::Read();
    midi_out.Send(byte);
  }
}

/* static */
void Dev::SendNote(uint8_t note, uint8_t velocity, uint8_t channel) {
  if (velocity > 0) {
    Send3(0x90 | channel, note, velocity);
    sent_notes_stack_.NoteOn(note, velocity, channel);
  } else {
    Send3(0x80 | channel, note, 0);
    sent_notes_stack_.NoteOff(note, channel);
  }
}

/* static */
void Dev::SendLater(uint8_t note, uint8_t velocity, uint8_t when) {
  event_scheduler.Schedule(note, velocity, when, channel_);
  if (!clock.running()) {
    clock.Start();
  }
}

/* static */
void Dev::SendScheduledNotes() {
  if (event_scheduler.size() == 0)
    return;

  uint8_t current = event_scheduler.root();
  while (current) {
    const SchedulerEntry& entry = event_scheduler.entry(current);
    if (entry.when) {
      break;
    }
    if (entry.note != kZombieSlot) {
      SendNote(entry.note, entry.velocity, entry.channel);
    }
    current = entry.next;
  }
  event_scheduler.Tick();
}

/* static */
void Dev::FlushQueue() {
  while (event_scheduler.size()) {
    SendScheduledNotes();
  }
}

/* static */
void Dev::SendViewSeqStep(uint8_t step) {
  if (!running()) {
    set_step(step);
    set_manual_step_selected(1); // Should be called before SendStep() to ignore step mute and legato
    SendStep(view_seq_, step);
  } else
  if (step_hold_) {
    set_step(step);
  }
}

/* static */
void Dev::SendSeqProgramChange() {
  if (flags_ & kSendBankSelectMsb) {
    Send3(0xb0 | channel_, midi::kBankMsb, bank_select_msb_);
  }

  if (flags_ & kSendBankSelectLsb) {
    Send3(0xb0 | channel_, midi::kBankLsb, bank_select_lsb_);
  }

  if (flags_ & kSendProgChange) {
    Send2(0xc0 | channel_, program_change_);
  }
}

///////////////////////////////////////////////////////////////////////////////
// Control change handling

/* static */
uint8_t Dev::HandleCC(uint8_t channel, uint8_t controller, uint8_t value) {
  if (!(ctrl_change_mode_ & CONTROL_CHANGE_RECV))
    return 0;

  int16_t setting;
  switch (controller) {

    case kCCSetSequence:
      if (value < kNumSeqs) {
        SetSeq(value);
      }
      break;

    case kCCSetDirection:
      if (value >= kMinSeqDirection && value <= kMaxSeqDirection) {
        set_seq_direction(value);
      }
      break;

    case kCCSetLinkMode:
      if (value >= kMinSeqLinkMode && value <= kMaxSeqLinkMode) {
        set_seq_link_mode(value);
      }
      break;

    case kCCSetStepLength:
      if (value >= kMinStepLength && value <= kMaxStepLength) {
        set_step_length(value);
      }
      break;

    case kCCIncreaseTempo:
      if (value) {
        setting = static_cast<int16_t>(tempo()) + 1;
        goto SetTempo;
      }
      break;

    case kCCDecreaseTempo:
      if (value) {
        setting = static_cast<int16_t>(tempo()) - 1;
        goto SetTempo;
      }
      break;

    case kCCSetTempo:
      setting = value * 2;
SetTempo: set_tempo(Clamp(setting, kMinTempo, kMaxTempo));
      break;

    case kCCToggleRun:
      if (value) {
        ToggleRun();
      }
      break;

    case kCCStopSequencer:
      if (value) {
        Stop();
      }
      break;

    case kCCStartSequencer:
      if (value) {
        Start();
      }
      break;

    case kCCToggleRecording:
      if (value) {
        ToggleRecording();
      }
      break;

    case kCCStopRecording:
      if (value) {
        StopRecording();
      }
      break;

    case kCCStartRecording:
      if (value) {
        StartRecording();
      }
      break;

    case kCCSetSeqSwitchMode:
      if (value >= kMinSeqSwitchMode && value <= kMaxSeqSwitchMode) {
        set_seq_switch_mode(value);
      }
      break;

    default:
      return 0;
  }

  ui.request_redraw();

  return 1;
}

///////////////////////////////////////////////////////////////////////////////
// Command handlers

/* static */
void Dev::CopySeq(uint8_t from, uint8_t to) {
  if (from != to) {
    seq_[to] = seq_[from];
  }
}

/* static */
void Dev::RotateSeq(uint8_t seq, uint8_t step) {
  uint8_t seq_length, rotate_count;
  Seq::Step* pSteps;

  switch (seq_link_mode_) {
    case SEQ_LINK_MODE_NONE:
      { seq_length = kNumSteps;
        rotate_count = seq_length - step;
        pSteps = seq_[seq].steps_ptr();
Rotate: Seq::ReverseSteps(pSteps, 0, seq_length - 1);
        Seq::ReverseSteps(pSteps, 0, rotate_count - 1);
        Seq::ReverseSteps(pSteps, rotate_count, seq_length - 1);
      }
      break;
    case SEQ_LINK_MODE_16:
      { step+= (seq & 1) * kNumSteps;
        seq_length = 2 * kNumSteps;
        rotate_count = seq_length - step;
        pSteps = seq_[seq & 2].steps_ptr();
        goto Rotate;
      }
      break;
    case SEQ_LINK_MODE_32:
      { step+= seq * kNumSteps;
        seq_length = 4 * kNumSteps;
        rotate_count = seq_length - step;
        pSteps = seq_[0].steps_ptr();
        goto Rotate;
      }
      break;
  }
}

/* static */
void Dev::RandomizeSeq(uint8_t seq, uint8_t scale) {
  switch (seq_link_mode_) {
    case SEQ_LINK_MODE_NONE:
      for (uint8_t n = 0; n < kNumSteps; n++) {
        set_seq_step_note(seq, n, GetRandomNote(scale));
      }
      break;
    case SEQ_LINK_MODE_16:
      for (uint8_t n = 0; n < kNumSteps; n++) {
        set_seq_step_note((seq & 2) + 0, n, GetRandomNote(scale));
        set_seq_step_note((seq & 2) + 1, n, GetRandomNote(scale));
      }
      break;
    case SEQ_LINK_MODE_32:
      for (uint8_t n = 0; n < kNumSteps; n++) {
        set_seq_step_note(0, n, GetRandomNote(scale));
        set_seq_step_note(1, n, GetRandomNote(scale));
        set_seq_step_note(2, n, GetRandomNote(scale));
        set_seq_step_note(3, n, GetRandomNote(scale));
      }
      break;
  }
}

/* static */
void Dev::RandomizeSeqStep(uint8_t seq, uint8_t step, uint8_t scale) {
  uint8_t note = GetRandomNote(scale);
  set_seq_step_note(seq, step, note);
}

/* static */
uint8_t Dev::GetRandomNote(uint8_t scale) {
  uint8_t note = seq_root_note_;

  // Normalize note
  uint8_t octave = 0;
  while(note >= 12) {
    note -= 12;
    ++octave;
  }

  // Get random note
  uint8_t random_note;
  do  {
    random_note = Scale::GetScaledNote(scale, Random::GetByte() & 0x7f);
    while(random_note >= 12) {
      random_note -= 12;
    }
  } while (random_note == note);

  // Up/down an octave
  static const uint8_t change_probability = 64;
  uint8_t random_byte = Random::GetByte();
  if (random_byte < change_probability) {
    if (octave > 2) octave--;
  } else
  if (random_byte > 256 - change_probability) {
    if (octave < 8) octave++;
  }

  // Restore octave
  while (octave > 0) {
    random_note += 12;
    --octave;
  }

  return random_note;
}

/* static */
uint8_t Dev::Randomize7F(uint8_t value, uint8_t randomize) {
  if (!randomize) return value;

  int16_t rand = (static_cast<int16_t>(Random::GetByte()) * (2 * randomize + 1)) / 256 - randomize;

  return Clamp7F(static_cast<int16_t>(value) + rand);
}

/* static */
uint8_t Dev::RandomizeNote(uint8_t note, uint8_t randomize, uint8_t scale) {
  if (!randomize) return note;

  int16_t transposition = (static_cast<int16_t>(Random::GetByte()) * (2 * randomize + 1)) / 256 - randomize;
  if (scale > 0) {
    uint8_t up = 1;
    if (transposition < 0) {
      transposition = - transposition;
      up = 0;
    }
    for (uint8_t n = 0; n < transposition; n++) {
      note = Scale::GetNextScaledNote(scale - 1, note, up);
    }
  } else {
    note = Clamp7F(static_cast<int16_t>(note) + transposition);
  }

  return note;
}

/* static */
void Dev::InitSeq(uint8_t seq) {
  switch (seq_link_mode_) {
    case SEQ_LINK_MODE_NONE:
      seq_[seq].Init();
      break;
    case SEQ_LINK_MODE_16:
      seq_[(seq & 2) + 0].Init();
      seq_[(seq & 2) + 1].Init();
      break;
    case SEQ_LINK_MODE_32:
      for (uint8_t n = 0; n < kNumSeqs; n++) {
        seq_[n].Init();
      }
      break;
  }
}

/* static */
void Dev::InitSeqStep(uint8_t seq, uint8_t step) {
  seq_[seq].InitStep(step);
}

///////////////////////////////////////////////////////////////////////////////
// Miscellaneous

/* static */
void Dev::SetChannel(uint8_t channel) {
  if (channel != channel_) {
    set_channel(channel);
    SilenceSentNotes();
    note_stack_.Clear();
  }
}

/* static */
void Dev::SetSongStep(uint8_t step) {
  song_step_ = step;
  song_step_play_count_ = 0;
  set_play_seq(song_step_seq(step));
  set_transpose_note(song_step_root_note(step));
}

/* static */
void Dev::SilenceSentNotes() {
  if (!sent_notes_stack_.size())
    return;

  for (uint8_t n = 0; n < sent_notes_stack_.size(); n++) {
    const NoteEntry& noteEntry = sent_notes_stack_.played_note(n);
    Send3(0x80 | noteEntry.channel, noteEntry.note, noteEntry.velocity);
  }

  sent_notes_stack_.Clear();
}

/* static */
void Dev::SilenceAllNotes() {
  Send3(0xb0 | channel_, 120, 0);
  Send3(0xb0 | channel_, 123, 0);
}

}  // namespace midialf
