// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Device model declaration.

#ifndef MIDIALF_DEV_H_
#define MIDIALF_DEV_H_

#include "avrlib/base.h"

#include "midialf/midialf.h"
#include "midialf/midi_out.h"
#include "midialf/note_stack.h"
#include "midialf/slot.h"
#include "midialf/song.h"
#include "midialf/seq.h"
#ifdef ENABLE_CV_EXT
#include "midialf/cv/cv.h"
#include "midialf/cv/port.h"
#endif

namespace midialf {

typedef uint8_t (*NoteCallback)(uint8_t channel, uint8_t note, uint8_t velocity);
typedef uint8_t (*ControlChangeCallback)(uint8_t channel, uint8_t controller, uint8_t value);
typedef void (*RunningCallback)();
typedef void (*TickCallback)();
typedef void (*StepCallback)();

class Dev {
 public:
  Dev() {}
  
  static void Init();
  
  // MIDI event handlers
  static void OnNoteOn(uint8_t channel, uint8_t note, uint8_t velocity);
  static void OnNoteOff(uint8_t channel, uint8_t note, uint8_t velocity);
  static void OnAftertouch(uint8_t channel, uint8_t note, uint8_t velocity);
  static void OnAftertouch(uint8_t channel, uint8_t velocity);
  static void OnControlChange(uint8_t channel, uint8_t controller, uint8_t value);
  static void OnProgramChange(uint8_t channel, uint8_t program);
  static void OnPitchBend(uint8_t channel, uint16_t pitch_bend);
  static void OnSysExByte(uint8_t sysex_byte);
  static void OnClock();
  static void OnStart();
  static void OnContinue();
  static void OnStop();

  static uint8_t CheckChannel(uint8_t channel);
  
  static void OnRawByte(uint8_t byte);

  static void OnRawMidiData(
    uint8_t status,
    uint8_t* data,
    uint8_t data_size,
    uint8_t accepted_channel);
  
  static void OnInternalClockTick();

  // Sequence data accessors

  static void get_name(uint8_t* name) { memcpy(name, name_, kNameLength); }
  static void set_name(const uint8_t* name) { memcpy(name_, name, kNameLength); }
  static uint8_t* name_ptr() { return name_; }

  #define DEFINE_ACCESSORS_(type, name) \
    static type name() { return name##_; } \
    static void set_##name(type name) { name##_ = name; } 

  #define DEFINE_ACCESSORS2(type, name, call) \
    static type name() { return name##_; } \
    static void set_##name(type name) { name##_ = name; call(); } 

  DEFINE_ACCESSORS2(uint8_t, slot, UpdateSlot)
  DEFINE_ACCESSORS_(uint8_t, channel)
  DEFINE_ACCESSORS2(uint8_t, tempo, UpdateClock)
  DEFINE_ACCESSORS2(uint8_t, step_length, UpdateClockPrescaler)
  DEFINE_ACCESSORS2(int16_t, swing, UpdateClock)
  DEFINE_ACCESSORS_(uint8_t, seq_root_note)
  DEFINE_ACCESSORS_(uint8_t, seq_link_mode)
  DEFINE_ACCESSORS_(uint8_t, seq_switch_mode)
  DEFINE_ACCESSORS_(uint8_t, seq_transpose_mode)
  DEFINE_ACCESSORS_(uint8_t, seq_direction)
  DEFINE_ACCESSORS_(uint8_t, steps_forward)
  DEFINE_ACCESSORS_(uint8_t, steps_backward)
  DEFINE_ACCESSORS_(uint8_t, steps_replay)
  DEFINE_ACCESSORS_(uint8_t, steps_interval)
  DEFINE_ACCESSORS_(uint8_t, steps_repeat)
  DEFINE_ACCESSORS_(uint8_t, steps_skip)
  DEFINE_ACCESSORS_(uint8_t, cc1_number)
  DEFINE_ACCESSORS_(uint8_t, cc2_number)
  DEFINE_ACCESSORS_(uint8_t, prog_change_mode)
  DEFINE_ACCESSORS_(uint8_t, ctrl_change_mode)
  DEFINE_ACCESSORS_(uint8_t, clock_mode)
  DEFINE_ACCESSORS_(uint8_t, bank_select_msb)
  DEFINE_ACCESSORS_(uint8_t, bank_select_lsb)
  DEFINE_ACCESSORS_(uint8_t, program_change)
  DEFINE_ACCESSORS_(uint16_t, flags)

  // Clock mode accessors

  static uint8_t InternalClock() { return !ExternalClock(); }
  static uint8_t ExternalClock() { return clock_mode_ & CLOCK_MODE_EXTERNAL; }
  static uint8_t SendOutputClock() { return clock_mode_ & CLOCK_MODE_SENDOUTPUT; }
  static uint8_t ContiguousClock() { return clock_mode_ & CLOCK_MODE_CONTIGUOUS; }

  // CV extension board accessors

#ifdef ENABLE_CV_EXT
  DEFINE_ACCESSORS2(uint8_t, strobe_width, port.UpdateStrobeWidth)

  static uint8_t cv_mode(uint8_t cv) { return cv_mode_[cv] & CVMODE_MASK; }
  static void set_cv_mode(uint8_t cv, uint8_t value) { cv_mode_[cv] = (value & CVMODE_MASK) | (cv_mode_[cv] & CVMODE_OFFSET); }

  static uint8_t cv_mode_offset(uint8_t cv) { return cv_mode_[cv] & CVMODE_OFFSET; }
  static void set_cv_mode_offset(uint8_t cv, uint8_t value) { SETFLAGTO(cv_mode_[cv], CVMODE_OFFSET, value); port.UpdateCvOffset(); }

  static uint8_t gate_mode(uint8_t gate) { return gate_mode_[gate] & GATEMODE_MASK; }
  static void set_gate_mode(uint8_t gate, uint8_t value) { gate_mode_[gate] = (value & GATEMODE_MASK) | (gate_mode_[gate] & GATEMODE_INVERT); }

  static uint8_t gate_mode_invert(uint8_t gate) { return gate_mode_[gate] & GATEMODE_INVERT; }
  static void set_gate_mode_invert(uint8_t gate, uint8_t value) { SETFLAGTO(gate_mode_[gate], GATEMODE_INVERT, value); port.UpdateGateInvert(); }
#endif

  #undef DEFINE_ACCESSORS_
  #undef DEFINE_ACCESSORS2

  // Sequence step data accessors

  #define DEFINE_SEQ_STEP_ACCESSORS(type, name) \
    static type view_seq_step_##name(uint8_t step) { return seq_[view_seq_].step_##name(step); } \
    static type play_seq_step_##name(uint8_t step) { return seq_[play_seq_].step_##name(step); } \
    static type seq_step_##name(uint8_t seq, uint8_t step) { return seq_[seq].step_##name(step); } \
    static void set_view_seq_step_##name(uint8_t step, type name) { seq_[view_seq_].set_step_##name(step, name); } \
    static void set_play_seq_step_##name(uint8_t step, type name) { seq_[play_seq_].set_step_##name(step, name); } \
    static void set_seq_step_##name(uint8_t seq, uint8_t step, type name) { seq_[seq].set_step_##name(step, name); }

  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, note)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, velocity)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, gate)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, cc1)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, cc2)
  DEFINE_SEQ_STEP_ACCESSORS(uint16_t, flags)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, mute)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, skip)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, legato)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, send_cc1)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, send_cc2)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, condition)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, cc1_condition)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, cc2_condition)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, randomize_note)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, randomize_note_scale)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, randomize_velocity)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, randomize_cc1)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, randomize_cc2)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, retrigger)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, retrigger_velocity)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, retrigger_transposition)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, retrigger_transposition_scale)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, retrigger_condition)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, has_condition)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, has_cc1_condition)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, has_cc2_condition)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, has_retrigger)
  DEFINE_SEQ_STEP_ACCESSORS(uint8_t, has_retrigger_condition)

  #undef DEFINE_SEQ_STEP_ACCESSORS

  static uint8_t all_steps_skipped() { return all_steps_skipped(play_seq_); }
  static uint8_t all_steps_skipped(uint8_t seq) { return seq_[seq].all_steps_skipped(); }

  static uint8_t non_skipped_steps_count() { return non_skipped_steps_count(play_seq_); }
  static uint8_t non_skipped_steps_count(uint8_t seq) { return seq_[seq].non_skipped_steps_count(); }

  // Song step accessors

  #define DEFINE_SONG_STEP_ACCESSORS(type, name) \
    static type song_step_##name(uint8_t step) { return song_.step_##name(step); } \
    static void set_song_step_##name(uint8_t step, uint8_t name) { song_.set_step_##name(step, name); }

  DEFINE_SONG_STEP_ACCESSORS(uint8_t, seq)
  DEFINE_SONG_STEP_ACCESSORS(uint8_t, play_count)
  DEFINE_SONG_STEP_ACCESSORS(uint8_t, root_note)

  #undef DEFINE_SONG_STEP_ACCESSORS

  // Runtime data accessors

  #define DEFINE_ACCESSOR_(type, name) \
    static type name() { return name##_; }

  #define DEFINE_ACCESSORS(type, name) \
    static type name() { return name##_; } \
    static void set_##name(type name) { name##_ = name; } 

  DEFINE_ACCESSOR_(uint8_t, running)
  DEFINE_ACCESSOR_(uint8_t, recording)
  DEFINE_ACCESSORS(uint8_t, view_seq)
  DEFINE_ACCESSORS(uint8_t, play_seq)
  DEFINE_ACCESSORS(uint8_t, step)
  DEFINE_ACCESSORS(uint8_t, transpose_note)
  DEFINE_ACCESSOR_(uint8_t, request_set_seq)
  DEFINE_ACCESSORS(uint8_t, manual_step_selected)
  DEFINE_ACCESSOR_(uint8_t, clock_prescaler)
  DEFINE_ACCESSOR_(uint8_t, song_step)

  #undef DEFINE_ACCESSOR_
  #undef DEFINE_ACCESSORS

  static uint8_t song_mode_active() { return flags_ & kSongModeActive; }
  static void set_song_mode_active(uint8_t active) { SETFLAGTO(flags_, kSongModeActive, active); }

  // Callbacks

  #define DEFINE_CALLBACK(type, name) \
    static type set_##name(type callback) { \
      type prev = name##_; \
      name##_ = callback; \
      return prev; \
    }

  DEFINE_CALLBACK(NoteCallback, note_callback)
  DEFINE_CALLBACK(ControlChangeCallback, control_change_callback)
  DEFINE_CALLBACK(RunningCallback, running_callback)
  DEFINE_CALLBACK(TickCallback, tick_callback)
  DEFINE_CALLBACK(StepCallback, step_callback)

  #undef DEFINE_CALLBACK

  // Output helpers

  static void SendNow(uint8_t byte);
  static void Send(uint8_t status, uint8_t* data, uint8_t size);
  static void Send2(uint8_t a, uint8_t b);
  static void Send3(uint8_t a, uint8_t b, uint8_t c);
  static void FlushOutputBuffer(uint8_t size);
  static void SendNote(uint8_t note, uint8_t velocity, uint8_t channel);
  static void SendLater(uint8_t note, uint8_t velocity, uint8_t when);
  static void SendScheduledNotes();
  static void FlushQueue();

  static void SendViewSeqStep(uint8_t step);

  // Operations

  static void InitDev();
  static void InitSlots();
  static void InitSlot();
  static void LoadSlot(uint8_t slot);
  static void SaveSlot(uint8_t slot);
  static void SaveSlot() {
    SaveSlot(slot_);
  }

  static void FromSlot(const Slot& slot);
  static void ToSlot(Slot& slot);

  static void SetPlaySeq(uint8_t seq);
  static void SetViewSeq(uint8_t seq) {
    set_view_seq(seq);
  }

  static void SetSeq(uint8_t seq) {
    SetPlaySeq(seq);
    SetViewSeq(seq);
  }

  static void ToggleRun();
  static void Start();
  static void Stop();

  static void StopRecording();
  static void StartRecording();
  static void ToggleRecording();

  static void ToggleStepHold() {
    step_hold_ = !step_hold_;
  }

  static uint8_t IsFirstStep(uint8_t step);
  static uint8_t IsFirstStep() {
    return IsFirstStep(step_);
  }

  static void CopySeq(uint8_t from, uint8_t to);
  static void RotateSeq(uint8_t seq, uint8_t step);
  static void RandomizeSeq(uint8_t seq, uint8_t scale);
  static void RandomizeSeqStep(uint8_t seq, uint8_t step, uint8_t scale);
  static uint8_t GetRandomNote(uint8_t scale);
  static uint8_t Randomize7F(uint8_t value, uint8_t randomize);
  static uint8_t RandomizeNote(uint8_t note, uint8_t randomize, uint8_t scale);

  static void InitSeq(uint8_t seq);
  static void InitSeqStep(uint8_t seq, uint8_t step);

  static void SetChannel(uint8_t channel);

  static void SetSongStep(uint8_t step);

  static void SendSeqProgramChange();

  static void SilenceSentNotes();
  static void SilenceAllNotes();

 private:
  static void Tick();
  static void AdvanceStep();
  static void AdvanceStep(uint8_t direction);
  static void SendStep(uint8_t seq, uint8_t step);
  static void SendStepCC(uint8_t seq, uint8_t step);
  static void SendStepRetrigger(uint8_t seq, uint8_t step, uint8_t note, uint8_t velocity);

  static void UpdateSlot();
  static void UpdateClock();
  static void UpdateClockPrescaler();

  static void RecordStep(uint8_t note, uint8_t velocity);
  static uint8_t AreAllStepsSkipped();
  static uint8_t NonSkippedStepsCount();
  static uint8_t GetRunningDirection(uint8_t reversed);

  static uint8_t CheckNoteCondition(uint8_t seq, uint8_t step);
  static uint8_t CheckCC1Condition(uint8_t seq, uint8_t step);
  static uint8_t CheckCC2Condition(uint8_t seq, uint8_t step);
  static uint8_t CheckRetriggerCondition(uint8_t seq, uint8_t step);

  static uint8_t HandleCC(uint8_t channel, uint8_t controller, uint8_t value);

  static const uint8_t noNote = 0xff;
  static const uint8_t noStep = 0xff;

  // Persistent data
  static uint8_t slot_;
  static uint8_t name_[kNameLength];
  static uint8_t channel_;
  static uint8_t tempo_;
  static uint8_t step_length_;
  static int16_t swing_;
  static uint8_t seq_root_note_;
  static uint8_t seq_link_mode_;
  static uint8_t seq_switch_mode_;
  static uint8_t seq_transpose_mode_;
  static uint8_t seq_direction_;
  static uint8_t steps_forward_;
  static uint8_t steps_backward_;
  static uint8_t steps_replay_;
  static uint8_t steps_interval_;
  static uint8_t steps_repeat_;
  static uint8_t steps_skip_;
  static uint8_t cc1_number_;
  static uint8_t cc2_number_;
  static uint8_t prog_change_mode_;
  static uint8_t ctrl_change_mode_;
  static uint8_t clock_mode_;
  static uint8_t bank_select_msb_;
  static uint8_t bank_select_lsb_;
  static uint8_t program_change_;
  static uint16_t flags_;
#ifdef ENABLE_CV_EXT
  static uint8_t cv_mode_[4];
  static uint8_t gate_mode_[4];
  static uint8_t strobe_width_;
#endif

  // Runtime data
  static uint8_t tick_;
  static uint8_t running_;
  static uint8_t recording_;
  static uint8_t clock_prescaler_;

  static uint8_t view_seq_;
  static uint8_t play_seq_;
  static uint8_t step_;
  static uint8_t step_hold_;  
  static uint8_t transpose_note_;
  static uint8_t pendulum_backward_;
  static uint8_t steps_forward_counter_;
  static uint8_t steps_replay_counter_;
  static uint8_t steps_interval_counter_;
  static uint8_t steps_repeat_counter_;
  static uint8_t steps_skip_counter_;
  static uint8_t saved_replay_step_;
  static uint8_t playing_steps_count_;
  static uint8_t request_set_seq_;
  static uint8_t next_step_;
  static uint8_t next_seq_;
  static uint8_t manual_step_selected_;
  static uint8_t last_recorded_step_;
  static uint8_t last_recorded_seq_;
  static uint8_t last_legato_note_;
  static uint8_t song_step_;
  static uint8_t song_step_play_count_;

  static Seq seq_[kNumSeqs];

  static Song song_;

  static NoteStack<16> note_stack_;
  static NoteStack<64> sent_notes_stack_;
  
  static NoteCallback note_callback_;
  static ControlChangeCallback control_change_callback_;
  static RunningCallback running_callback_;
  static TickCallback tick_callback_;
  static StepCallback step_callback_;

  DISALLOW_COPY_AND_ASSIGN(Dev);
};

extern Dev dev;

}  // namespace midialf

#endif // MIDIALF_DEV_H_
