// Copyright 2012 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// Based on MIDIPal code by Olivier Gillet (ol.gillet@gmail.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// CC LFO class.

#include "midialf/lfo.h"
#include "midialf/duration.h"
#include "midialf/clock.h"
#include "midialf/ui.h"
#ifdef ENABLE_CV_EXT
#include "midialf/cv/cv.h"
#include "midialf/cv/port.h"
#endif

#include "midi/midi.h"

#include "avrlib/op.h"
#include "avrlib/random.h"

namespace midialf {

using namespace avrlib;

/* extern */
Lfo lfo;

/* <static> */
uint8_t Lfo::resolution_ = kDefLfoResolution;
LfoSettings Lfo::lfo_[kNumLfos];

uint16_t Lfo::phase_[kNumLfos];
uint16_t Lfo::phase_increment_[kNumLfos];

uint8_t Lfo::tick_;
uint8_t Lfo::midi_clock_prescaler_;
uint8_t Lfo::running_;
/* </static> */

void LfoSettings::Init(uint8_t lfo) {
  amount = kDefLfoAmount;
  center = kDefLfoCenter;
  waveform = kDefLfoWaveform;
  rate = kDefLfoRate;
  sync = kDefLfoSync;

  switch (lfo) {
    case 0:
      cc_number = midi::kVolume;
      break;
    case 1:
      cc_number = midi::kPan;
      break;
  }
}

#define FIX_DEF(name, Name) \
  if (name < kMin##Name || name > kMax##Name) { \
    name = kDef##Name; \
  }

void LfoSettings::Validate() {
  FIX_DEF(waveform, LfoWaveform);
  FIX_DEF(cc_number, LfoCCNumber);
  FIX_DEF(amount, LfoAmount);
  FIX_DEF(center, LfoCenter);
  FIX_DEF(rate, LfoRate);
  FIX_DEF(sync, LfoSync);
}

#undef FIX_DEF

/* static */
void Lfo::Init() {
  for (uint8_t i = 0; i < kNumLfos; ++i) {
    lfo_[i].Init(i);
  }
}

/* static */
void Lfo::UpdatePrescaler() {
  midi_clock_prescaler_ = Duration::GetMidiClockTicks(resolution_);
  uint16_t factor = midi_clock_prescaler_;
  for (uint8_t i = 0; i < kNumLfos; ++i) {
    phase_increment_[i] = ResourcesManager::Lookup<uint16_t, uint8_t>(
      lut_res_lfo_increments, lfo_[i].rate) * factor;
  }
}

/* static */
void Lfo::OnStep() {
  // Sync lfos
  for (uint8_t i = 0; i < kNumLfos; ++i) {
    if (lfo_[i].sync == LFO_SYNC_STEP || (lfo_[i].sync == LFO_SYNC_SEQSTART && dev.IsFirstStep())) {
      phase_[i] = 0;
    }
  }
}

/* static */
void Lfo::Stop() {
  if (!running_)
    return;

  running_ = 0;
}

/* static */
void Lfo::Start() {
  if (running_)
    return;

  tick_ = midi_clock_prescaler_ - 1;
  running_ = 1;

  // Sync lfos
  for (uint8_t i = 0; i < kNumLfos; ++i) {
    if (lfo_[i].sync == LFO_SYNC_START) {
      phase_[i] = 0;
    }
  }
}

/* static */
void Lfo::Tick() {
  ++tick_;
  if (tick_ >= midi_clock_prescaler_) {
    tick_ = 0;
    for (uint8_t i = 0; i < kNumLfos; ++i) {
      // Check if no waveform and skip
      if (lfo_[i].waveform == 0)
        continue;

#ifdef ENABLE_CV_EXT
      if (phase_[i] < phase_increment_[i]) {
        port.SendLFO(i);
      }
#endif
      // Calculate phase value
      uint8_t value;
      uint8_t skip = 0;
      if (lfo_[i].waveform == kNumLfoWaveforms - 1) {
        if (phase_[i] < phase_increment_[i]) {
          value = Random::GetByte();
        } else {
          skip = 1;
        }
      } else {
        uint16_t offset = U8U8Mul(lfo_[i].waveform - 1, 129);
        value = InterpolateSample(
            wav_res_lfo_waveforms + offset,
            phase_[i] >> 1);
      }

      phase_[i] += phase_increment_[i];

      if (!skip) {
        int16_t scaled_value = static_cast<int16_t>(lfo_[i].center) + 
            S8S8MulShift8((lfo_[i].amount - 63) << 1, value - 128);
        scaled_value = Clip(scaled_value, 0, 127);
        if (lfo_[i].cc_number) {
          dev.Send3(
              0xb0 | dev.channel(),
              lfo_[i].cc_number,
              scaled_value & 0x7f);
        }
#ifdef ENABLE_CV_EXT
        switch (i) {
          case 0: cv.SendLFO1(scaled_value & 0x7f); break;
          case 1: cv.SendLFO2(scaled_value & 0x7f); break;
        }
#endif
      }
    }
  }
}

} // namespace midialf
