// Copyright 2012 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// Based on MIDIPal code by Olivier Gillet (ol.gillet@gmail.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// CC LFO class.

#ifndef MIDIALF_LFO_H_
#define MIDIALF_LFO_H_

#include "midialf/midialf.h"

namespace midialf {

struct LfoSettings {
  LfoSettings() {}

  void Init(uint8_t lfo);
  void Validate();

  uint8_t waveform;
  uint8_t cc_number;
  uint8_t amount;
  uint8_t center;
  uint8_t rate;
  uint8_t sync;
};

const uint8_t kNumLfos = 2;
const uint8_t kNumLfoWaveforms = 19;
const uint8_t kNumLfoRates = 18;

enum LfoSync {
  LFO_SYNC_FREE_RUNNING,
  LFO_SYNC_START,
  LFO_SYNC_SEQSTART,
  LFO_SYNC_STEP,
};

const uint8_t kMinLfoWaveform = 0;
const uint8_t kMaxLfoWaveform = kNumLfoWaveforms - 1;
const uint8_t kDefLfoWaveform = 0;  // none

const uint8_t kMinLfoCCNumber = 0;
const uint8_t kMaxLfoCCNumber = 119;
const uint8_t kDefLfoCCNumber = 0;

const uint8_t kMinLfoAmount = 0;
const uint8_t kMaxLfoAmount = 127;
const uint8_t kDefLfoAmount = 127;

const uint8_t kMinLfoCenter = 0;
const uint8_t kMaxLfoCenter = 127;
const uint8_t kDefLfoCenter = 63;

const uint8_t kMinLfoRate = 0;
const uint8_t kMaxLfoRate = kNumLfoRates - 1;
const uint8_t kDefLfoRate = 3;   // 1 Bar

const uint8_t kMinLfoSync = LFO_SYNC_FREE_RUNNING;
const uint8_t kMaxLfoSync = LFO_SYNC_STEP;
const uint8_t kDefLfoSync = LFO_SYNC_FREE_RUNNING;

const uint8_t kMinLfoResolution = 0;
const uint8_t kMaxLfoResolution = kNoteDurationCount - 1;
const uint8_t kDefLfoResolution = 0;

class Lfo {
 public:
  Lfo() { }

  static void Init();

  static void UpdatePrescaler();

  static void Stop();
  static void Start();
  static void Tick();

  static void OnStep();
  
  static uint8_t resolution() { return resolution_; }
  static void set_resolution(uint8_t resolution) { resolution_ = resolution; UpdatePrescaler(); }

  #define DEFINE_ACCESSORS(type, name) \
    static type name(uint8_t lfo) { return lfo_[lfo].name; } \
    static void set_##name(uint8_t lfo, type name) { lfo_[lfo].name = name; } 

  DEFINE_ACCESSORS(uint8_t, cc_number)
  DEFINE_ACCESSORS(uint8_t, amount)
  DEFINE_ACCESSORS(uint8_t, center)
  DEFINE_ACCESSORS(uint8_t, waveform)
  DEFINE_ACCESSORS(uint8_t, rate)
  DEFINE_ACCESSORS(uint8_t, sync)

  #undef DEFINE_ACCESSORS

  static void get_settings(uint8_t lfo, LfoSettings& settings) { settings = lfo_[lfo]; }
  static void set_settings(uint8_t lfo, const LfoSettings& settings) { lfo_[lfo] = settings; }

 private:

  static uint8_t resolution_;
  static LfoSettings lfo_[kNumLfos];

  static uint16_t phase_[kNumLfos];
  static uint16_t phase_increment_[kNumLfos];
  
  static uint8_t tick_;
  static uint8_t midi_clock_prescaler_;
  static uint8_t running_;
  
  DISALLOW_COPY_AND_ASSIGN(Lfo);
};

extern Lfo lfo;

} // namespace midialf

#endif // MIDIALF_LFO_H_
