// Copyright 2017 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

#include "avrlib/gpio.h"
#include "avrlib/boot.h"
#include "avrlib/time.h"
#include "avrlib/timer.h"
#include "avrlib/serial.h"
#include "avrlib/watchdog_timer.h"

#include "midi/midi.h"

#include "midialf/midialf.h"
#include "midialf/event_scheduler.h"
#include "midialf/midi_handler.h"
#include "midialf/midi_leds.h"
#include "midialf/midi_out.h"
#include "midialf/display.h"
#include "midialf/storage.h"
#include "midialf/clock.h"
#include "midialf/eemem.h"
#include "midialf/leds.h"
#include "midialf/dev.h"
#include "midialf/ui.h"

#ifdef ENABLE_CV_EXT
#include "midialf/cv/cv.h"
#include "midialf/cv/port.h"
#endif

#include <util/delay.h>

using namespace avrlib;
using namespace midi;
using namespace midialf;

Serial<MidiPort, 31250, POLLED, POLLED> midi_io;
MidiStreamParser<MidiHandler> midi_parser;

// Timer2 ISR: midi i/o and ui

inline void PollMidiIn() {
  // Receive midi input
  if (midi_io.readable()) {
    uint8_t byte = midi_io.ImmediateRead();
    if (byte != 0xfe) {
#ifndef ENABLE_CV_EXT
      if (byte != 0xf8) midi_leds.FlashMidiIn();
#endif
      midi_parser.PushByte(byte);
    }
  }
}

inline void SendMidiOut() {
  // Send midi output
  if (MidiHandler::OutputBuffer::readable() && midi_io.writable()) {
    uint8_t byte = MidiHandler::OutputBuffer::ImmediateRead();
#ifndef ENABLE_CV_EXT
    if (byte != 0xf8) midi_leds.FlashMidiOut();
#endif
    midi_io.Overwrite(byte);
  }
}

// Called at 4.9KHz

ISR(TIMER2_OVF_vect, ISR_NOBLOCK) {

  // Handle MIDI I/O
  PollMidiIn();
  SendMidiOut();

  // Handle CV/Gates
#ifdef ENABLE_CV_EXT
  cv.Tick();
  port.Tick();
#endif

  // Handle lower priority tasks
  static uint8_t sub_clock;
  ++sub_clock;
  if ((sub_clock & 1) == 0) {
    // 2.45KHz
    ui.Poll();
    if ((sub_clock & 3) == 0) {
      // 1.225KHz
      TickSystemClock();
#ifndef ENABLE_CV_EXT
      midi_leds.Tick();
#endif
      leds.Write();
      if ((sub_clock & 7) == 0) {
        // 306Hz
        display.BlinkCursor();
        ui.Tick();
      }
    }
  } 
}

// Timer1 ISR: internal clock

ISR(TIMER1_COMPA_vect) {
  PwmChannel1A::set_frequency(clock.Tick());
  if (clock.running()) {
    dev.OnInternalClockTick();
  }
}

void Init() {
  sei();
  UCSR0B = 0;

  event_scheduler.Init();

  Timer<1>::set_prescaler(1);
  Timer<1>::set_mode(0, _BV(WGM12), 3);
  PwmChannel1A::set_frequency(6510);
  Timer<1>::StartCompare();
  
  Timer<2>::set_prescaler(2);
  Timer<2>::set_mode(TIMER_PWM_PHASE_CORRECT);
  Timer<2>::Start();

#ifdef ENABLE_CV_EXT
  cv.Init();
  port.Init();
#else
  IOPort1::set_mode(DIGITAL_OUTPUT);
  IOPort2::set_mode(DIGITAL_OUTPUT);
#endif

  display.Init();
  storage.Init();
  midi_io.Init();
  midi_out.Init();
#ifndef ENABLE_CV_EXT
  midi_leds.Init();
#endif
  clock.Init();
  eemem.Init();
  leds.Init();
  dev.Init();
  ui.Init();
}

int main(void) {
  ResetWatchdog();
  Init();

  while (1) {
    ui.DoEvents();
  }
}
