// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.

#ifndef MIDIALF_MIDIALF_H_
#define MIDIALF_MIDIALF_H_

#include "avrlib/base.h"

#include "midialf/hardware_config.h"
#include "midialf/resources.h"
#include "midialf/settings.h"
#include "midialf/duration.h"
#include "midialf/scale.h"
#include "midialf/cond.h"

#include <string.h>

namespace midialf {

#define MIDIALF_VERSION "2.00"

// Configuration

static const uint8_t kNumSeqs = 4;
static const uint8_t kNumSteps = 8;
static const uint8_t kNumSlots = 64;

static const uint8_t kDefChannel = 0;

static const uint8_t kMinNameChar = ' ';
static const uint8_t kMaxNameChar = 0x7f;
static const uint8_t kNameLength = 16;

static const uint8_t kMinTempo = 25;
static const uint8_t kMaxTempo = 250;
static const uint8_t kDefTempo = 120;

static const uint8_t kMinClockMode = 0;
static const uint8_t kMaxClockMode = CLOCK_MODE_EXTERNAL | CLOCK_MODE_SENDOUTPUT | CLOCK_MODE_CONTIGUOUS;
static const uint8_t kDefClockMode = CLOCK_MODE_SENDOUTPUT;

static const uint8_t kMinStepLength = 0;
static const uint8_t kMaxStepLength = kNoteDurationCount - 1;
static const uint8_t kDefStepLength = k8thNote;

static const int16_t kMinSwing = 1;
static const int16_t kMaxSwing = 999;
static const int16_t kDefSwing = 500;

static const uint8_t kMinSeqRootNote = 0;
static const uint8_t kMaxSeqRootNote = 0x7f;
static const uint8_t kDefSeqRootNote = 60; // C4

static const uint8_t kMinSeqLinkMode = SEQ_LINK_MODE_NONE;
static const uint8_t kMaxSeqLinkMode = SEQ_LINK_MODE_32;
static const uint8_t kDefSeqLinkMode = SEQ_LINK_MODE_NONE;

static const uint8_t kMinSeqSwitchMode = SEQ_SWITCH_MODE_IMMEDIATE;
static const uint8_t kMaxSeqSwitchMode = SEQ_SWITCH_MODE_ONSEQEND;
static const uint8_t kDefSeqSwitchMode = SEQ_SWITCH_MODE_IMMEDIATE;

static const uint8_t kMinSeqTransposeMode = SEQ_TRANSPOSE_MODE_NONE;
static const uint8_t kMaxSeqTransposeMode = SEQ_TRANSPOSE_MODE_NOTE;
static const uint8_t kDefSeqTransposeMode = SEQ_TRANSPOSE_MODE_NOTE;

static const uint8_t kMinSeqDirection = SEQ_DIRECTION_FORWARD;
static const uint8_t kMaxSeqDirection = SEQ_DIRECTION_RANDOM;
static const uint8_t kDefSeqDirection = SEQ_DIRECTION_FORWARD;

static const uint8_t kMinStepCount = 0;
static const uint8_t kMaxStepCount = kNumSteps;
static const uint8_t kDefStepCount = 0;

static const uint8_t kMinCCNumber = 0;
static const uint8_t kMaxCCNumber = 119;
static const uint8_t kDefCC1Number = 7;  // Volume
static const uint8_t kDefCC2Number = 10; // Pan

static const uint8_t kMinProgChangeMode = PROGRAM_CHANGE_NONE;
static const uint8_t kMaxProgChangeMode = PROGRAM_CHANGE_BOTH;
static const uint8_t kDefProgChangeMode = PROGRAM_CHANGE_NONE;

static const uint8_t kMinCtrlChangeMode = CONTROL_CHANGE_NONE;
static const uint8_t kMaxCtrlChangeMode = CONTROL_CHANGE_BOTH;
static const uint8_t kDefCtrlChangeMode = CONTROL_CHANGE_NONE;

static const uint8_t kMinBankSelectMsb = 0;
static const uint8_t kMaxBankSelectMsb = 0x7f;
static const uint8_t kDefBankSelectMsb = 0;

static const uint8_t kMinBankSelectLsb = 0;
static const uint8_t kMaxBankSelectLsb = 0x7f;
static const uint8_t kDefBankSelectLsb = 0;

static const uint8_t kMinProgramChange = 0;
static const uint8_t kMaxProgramChange = 0x7f;
static const uint8_t kDefProgramChange = 0;

static const uint8_t kSendBankSelectMsb = 0x01;
static const uint8_t kSendBankSelectLsb = 0x02;
static const uint8_t kSendProgChange    = 0x04;
static const uint8_t kSongModeActive    = 0x08;
static const uint8_t kDefFlags = 0;

static const uint8_t kMinStepNote = 0;
static const uint8_t kMaxStepNote = 0x7f;
static const uint8_t kDefStepNote = 60; // C4

static const uint8_t kMinStepVelocity = 1;
static const uint8_t kMaxStepVelocity = 0x7f;
static const uint8_t kDefStepVelocity = 100;

static const uint8_t kMinStepGate = 0;
static const uint8_t kMaxStepGate = kNoteDurationCount - 1;
static const uint8_t kDefStepGate = k16thNote;

static const uint8_t kMinStepCC = 0;
static const uint8_t kMaxStepCC = 0x7f;
static const uint8_t kDefStepCC = 100;

static const uint8_t kMinStepCondition = kMinCondition;
static const uint8_t kMaxStepCondition = kMaxCondition;
static const uint8_t kDefStepCondition = kDefCondition;

static const uint8_t kMinStepCC1Condition = kMinCondition;
static const uint8_t kMaxStepCC1Condition = kMaxCondition;
static const uint8_t kDefStepCC1Condition = kDefCondition;

static const uint8_t kMinStepCC2Condition = kMinCondition;
static const uint8_t kMaxStepCC2Condition = kMaxCondition;
static const uint8_t kDefStepCC2Condition = kDefCondition;

static const uint8_t kMinStepRetrigger = 2;
static const uint8_t kMaxStepRetrigger = 16;
static const uint8_t kDefStepRetrigger = 2;

static const uint8_t kMinStepRetriggerVelocity = 0;
static const uint8_t kMaxStepRetriggerVelocity = 127;
static const uint8_t kDefStepRetriggerVelocity = 63;

static const uint8_t kMinStepRetriggerTransposition = 0;
static const uint8_t kMaxStepRetriggerTransposition = 48 * 2;
static const uint8_t kDefStepRetriggerTransposition = 48;

static const uint8_t kMinStepRetriggerTranspositionScale = 0;
static const uint8_t kMaxStepRetriggerTranspositionScale = kNumScales; // none + all scales
static const uint8_t kDefStepRetriggerTranspositionScale = 0;

static const uint8_t kMinStepRetriggerCondition = kMinCondition;
static const uint8_t kMaxStepRetriggerCondition = kMaxCondition;
static const uint8_t kDefStepRetriggerCondition = kDefCondition;

static const uint8_t kMinStepRandomizeNote = 0;
static const uint8_t kMaxStepRandomizeNote = 0x7f;
static const uint8_t kDefStepRandomizeNote = 0;

static const uint8_t kMinStepRandomizeNoteScale = 0;
static const uint8_t kMaxStepRandomizeNoteScale = kNumScales; // none + all scales
static const uint8_t kDefStepRandomizeNoteScale = 0;

static const uint8_t kMinStepRandomizeVelocity = 0;
static const uint8_t kMaxStepRandomizeVelocity = 0x7f;
static const uint8_t kDefStepRandomizeVelocity = 0;

static const uint8_t kMinStepRandomizeCC1 = 0;
static const uint8_t kMaxStepRandomizeCC1 = 0x7f;
static const uint8_t kDefStepRandomizeCC1 = 0;

static const uint8_t kMinStepRandomizeCC2 = 0;
static const uint8_t kMaxStepRandomizeCC2 = 0x7f;
static const uint8_t kDefStepRandomizeCC2 = 0;

static const uint8_t kMinSongSeq = 0;
static const uint8_t kMaxSongSeq = kNumSeqs - 1;
static const uint8_t kDefSongSeq = 0;

static const uint8_t kMinSongPlayCount = 0;
static const uint8_t kMaxSongPlayCount = 0x3f;
static const uint8_t kDefSongPlayCount = 0;

static const uint8_t kMinSongRootNote = kMinSeqRootNote;
static const uint8_t kMaxSongRootNote = kMaxSeqRootNote;
static const uint8_t kDefSongRootNote = kDefSeqRootNote;

static const uint8_t kMinCVMode = CVMODE_NOTE;
static const uint8_t kMaxCVMode = CVMODE_MAX;
static const uint8_t kDefCVMode = kMinCVMode;
static const uint8_t kDefCV1Mode = CVMODE_NOTE;
static const uint8_t kDefCV2Mode = CVMODE_VELO;
static const uint8_t kDefCV3Mode = CVMODE_CC1;
static const uint8_t kDefCV4Mode = CVMODE_CC2;

static const uint8_t kMinGateMode = GATEMODE_GATE;
static const uint8_t kMaxGateMode = GATEMODE_MAX;
static const uint8_t kDefGateMode = kMinGateMode;
static const uint8_t kDefGate1Mode = GATEMODE_GATE;
static const uint8_t kDefGate2Mode = GATEMODE_STROBE;
static const uint8_t kDefGate3Mode = GATEMODE_LFO1;
static const uint8_t kDefGate4Mode = GATEMODE_LFO2;

static const uint8_t kDefStrobeWidth = 0; // 1ms
static const uint8_t kMinStrobeWidth = 0;
static const uint8_t kMaxStrobeWidth = 4;

// Device control change events

static const uint8_t kCCSetSequence = 106;
static const uint8_t kCCSetDirection = 107;
static const uint8_t kCCSetLinkMode = 108;
static const uint8_t kCCSetStepLength = 109;
static const uint8_t kCCIncreaseTempo = 110;
static const uint8_t kCCDecreaseTempo = 111;
static const uint8_t kCCSetTempo = 112;
static const uint8_t kCCToggleRun = 113;
static const uint8_t kCCStopSequencer = 114;
static const uint8_t kCCStartSequencer = 115;
static const uint8_t kCCToggleRecording = 116;
static const uint8_t kCCStopRecording = 117;
static const uint8_t kCCStartRecording = 118;
static const uint8_t kCCSetSeqSwitchMode = 119;

// Useful declarations

#define numbof(a)  (sizeof(a)/sizeof(a[0]))
#define lengof(s)  (numbof(s) - 1)

#define max(a,b)  (((a) > (b)) ? (a) : (b))
#define min(a,b)  (((a) < (b)) ? (a) : (b))

#define SETFLAG(f,b)  (f)|= (b)
#define CLRFLAG(f,b)  (f)&=~(b)
#define SETFLAGTO(f,b,c)  if (c) SETFLAG(f,b); else CLRFLAG(f,b);

#define PSTRN(str) PSTR(str), lengof(str)

#ifdef _MSC_VER
#define OFFSETOF(s,m) (uint16_t)offsetof(s,m)
#else
#define OFFSETOF(s,m) (uint16_t)&reinterpret_cast<const volatile char&>((((s *)0)->m))
#endif

#ifndef _MSC_VER
#define assert(expr)  (void)0
#endif

// 50ns @20MHz
static inline void nop() { 
  __asm__ volatile (
    "nop"
    ); 
}

// count*160ns @20MHz, 0=38us
static inline void Delay(uint8_t count) {
  __asm__ volatile (
    "1: dec %0" "\n\t"
    "brne 1b"
    : "=r" (count)
    : "0" (count)
  );
}

// Disable interrupts
class DisableInterrupts {
  uint8_t sreg_;
 public:
   DisableInterrupts() { sreg_= SREG; cli(); }
   ~DisableInterrupts() { SREG = sreg_; }
};

#ifndef ENABLE_CV_EXT

// Profiling aids
template <typename IOPort>
struct RaiseIOPort {
  RaiseIOPort() { IOPort::High(); }
  RaiseIOPort(uint8_t count) { IOPort::High(); Delay(count); }
  ~RaiseIOPort() { IOPort::Low(); }
};

typedef RaiseIOPort<IOPort1> RaiseIOPort1;
typedef RaiseIOPort<IOPort2> RaiseIOPort2;

#endif // #ifndef ENABLE_CV_EXT

} // namespace midialf

#endif  // MIDIALF_MIDIALF_H_
