#!/usr/bin/python2.5
#
# Copyright 2015 Peter Kvitek.
#
# Author: Peter Kvitek (pete@kvitek.com)
#
# Based on code by: Olivier Gillet (ol.gillet@gmail.com)
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# -----------------------------------------------------------------------------
#
# Lookup table definitions.

import numpy
from scipy.interpolate import interp1d
import numpy as np

lookup_tables = []
lookup_tables_8 = []

"""----------------------------------------------------------------------------
Velocity and aftertouch curves
----------------------------------------------------------------------------"""

xCurve = np.linspace(0, 1.0, 128)

xSoft1 = np.array([0.0, 0.35, 0.50, 1.0])
ySoft1 = np.array([0.0, 0.50, 0.65, 1.0])
fSoft1 = interp1d(xSoft1, ySoft1, kind = 'cubic')
ySoft1Curve = fSoft1(xCurve)

xSoft2 = np.array([0.0, 0.25, 0.50, 1.0])
ySoft2 = np.array([0.0, 0.50, 0.75, 1.0])
fSoft2 = interp1d(xSoft2, ySoft2, kind = 'cubic')
ySoft2Curve = fSoft2(xCurve)

xSoft3 = np.array([0.0, 0.40, 0.60, 1.0])
ySoft3 = np.array([0.0, 0.50, 0.75, 1.0])
fSoft3 = interp1d(xSoft3, ySoft3, kind = 'cubic')
ySoft3Curve = fSoft3(xCurve)

xHard1 = np.array([0.0, 0.50, 0.65, 1.0])
yHard1 = np.array([0.0, 0.35, 0.50, 1.0])
fHard1 = interp1d(xHard1, yHard1, kind = 'cubic')
yHard1Curve = fHard1(xCurve)

xHard2 = np.array([0.0, 0.50, 0.75, 1.0])
yHard2 = np.array([0.0, 0.25, 0.50, 1.0])
fHard2 = interp1d(xHard2, yHard2, kind = 'cubic')
yHard2Curve = fHard2(xCurve)

xHard3 = np.array([0.0, 0.40, 0.60, 1.0])
yHard3 = np.array([0.0, 0.25, 0.50, 1.0])
fHard3 = interp1d(xHard3, yHard3, kind = 'cubic')
yHard3Curve = fHard3(xCurve)

xWide1 = np.array([0.0, 0.25, 0.75, 1.0])
yWide1 = np.array([0.0, 0.35, 0.65, 1.0])
fWide1 = interp1d(xWide1, yWide1, kind = 'cubic')
yWide1Curve = fWide1(xCurve)

xWide2 = np.array([0.0, 0.35, 0.65, 1.0])
yWide2 = np.array([0.0, 0.45, 0.55, 1.0])
fWide2 = interp1d(xWide2, yWide2, kind = 'cubic')
yWide2Curve = fWide2(xCurve)

lookup_tables_8.append(('curve_soft1', ySoft1Curve / ySoft1Curve.max() * 127.0))
lookup_tables_8.append(('curve_soft2', ySoft2Curve / ySoft2Curve.max() * 127.0))
lookup_tables_8.append(('curve_soft3', ySoft3Curve / ySoft3Curve.max() * 127.0))
lookup_tables_8.append(('curve_hard1', yHard1Curve / yHard1Curve.max() * 127.0))
lookup_tables_8.append(('curve_hard2', yHard2Curve / yHard2Curve.max() * 127.0))
lookup_tables_8.append(('curve_hard3', yHard3Curve / yHard3Curve.max() * 127.0))
lookup_tables_8.append(('curve_wide1', yWide1Curve / yWide1Curve.max() * 127.0))
lookup_tables_8.append(('curve_wide2', yWide2Curve / yWide2Curve.max() * 127.0))

"""----------------------------------------------------------------------------
Increments for LFOs
----------------------------------------------------------------------------"""

periods = [384, 288, 192, 144, 96, 72, 64, 48, 36, 32, 24, 16, 12, 8, 6, 4, 3, 2, 1]

lfo_increments = 65536 / numpy.array(periods)
lfo_increments[-1] = 65535

lookup_tables.append(('lfo_increments', lfo_increments))


"""-----------------------------------------------------------------------------
LFO waveshapes
----------------------------------------------------------------------------"""

def convolve(curve, impulse):
  return numpy.fft.irfft(numpy.fft.rfft(curve) * numpy.fft.rfft(impulse))


def scale(x, min=1, max=254, center=True, dithering=0):
  mx = x.max()
  mn = x.min()
  x = (x - mn) / (mx - mn)
  x = numpy.round(x * (max - min) + min)
  target_type = numpy.uint8
  x[x < numpy.iinfo(target_type).min] = numpy.iinfo(target_type).min
  x[x > numpy.iinfo(target_type).max] = numpy.iinfo(target_type).max
  return x.astype(target_type)


custom_lfos = []

t = numpy.arange(0, 128) / 128.0
ramp = t
triangle = 2 * t * (t < 0.5) + (2.0 - 2 * t) * (t >= 0.5)
square = (t < 0.5)
bipolar_triangle = 2 * triangle - 1.0
sine = numpy.sin(2 * numpy.pi * t)

# Sine and harmonics (4)
sine_2 = numpy.sin(2 * numpy.pi * t) + 0.7 * numpy.sin(4 * numpy.pi * t)
sine_3 = numpy.sin(2 * numpy.pi * t) + 0.7 * numpy.sin(6 * numpy.pi * t)
sine_5 = numpy.sin(2 * numpy.pi * t) + 0.7 * numpy.sin(10 * numpy.pi * t)
custom_lfos.append(triangle)
custom_lfos.append(square * 1.0)
custom_lfos.append(ramp)
custom_lfos.append(sine)
custom_lfos.append(sine_2)
custom_lfos.append(sine_3)
custom_lfos.append(sine_5)
# Gurgles (2)
window = (1.0 - numpy.cos(4 * numpy.pi * t)) / 2.0
custom_lfos.append(numpy.maximum(sine_2, 0.0) * window)
custom_lfos.append(numpy.maximum(sine_3, 0.0) * window)

# Bat (2)
for fold_amount in [0.5, 1.0]:
  fold = (1 + fold_amount) * bipolar_triangle
  fold[fold > 1.0] = 2.0 - fold[fold > 1.0]
  fold[fold < -1.0] = -2.0 - fold[fold < -1.0]
  custom_lfos.append(fold)

# Spiky (2)
spike = 2 ** (4 * triangle) - 1
bipolar_spike = numpy.sign(bipolar_triangle) * (
    2 ** (4 * numpy.abs(bipolar_triangle)) - 1)
custom_lfos.append(spike)
custom_lfos.append(bipolar_spike)

# Low-pass filtered ramp and square (2)
smooth = numpy.exp(-10 * t)
custom_lfos.append(convolve(ramp, smooth))
custom_lfos.append(convolve(square, smooth))

# Low-pass filtered ramp and square with resonance (2)
bouncy = numpy.exp(-10 * t) * numpy.sin(16 * numpy.pi * t)
custom_lfos.append(convolve(ramp, bouncy))
custom_lfos.append(convolve(square, bouncy))

lfo_waveforms = numpy.zeros((129 * len(custom_lfos),), dtype=numpy.uint8)
for i, values in enumerate(custom_lfos):
  values = scale(values)
  lfo_waveforms[i * 129: i * 129 + 128] = values
  lfo_waveforms[i * 129 + 128] = values[0]

waveforms = [('lfo_waveforms', lfo_waveforms)]

