// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Sequence model.

#include "midialf/midialf.h"
#include "midialf/seq.h"

using namespace avrlib;

namespace midialf {

void Seq::Step::Init() {
  note = kDefStepNote;
  velocity = kDefStepVelocity;
  gate = kDefStepGate;
  cc1 = kDefStepCC;
  cc2 = kDefStepCC;
  flags = 0;
  condition = kDefStepCondition;
  cc1_condition = kDefStepCC1Condition;
  cc2_condition = kDefStepCC2Condition;
  randomize_note = kDefStepRandomizeNote;
  randomize_note_scale = kDefStepRandomizeNoteScale;
  randomize_velocity = kDefStepRandomizeVelocity;
  randomize_cc1 = kDefStepRandomizeCC1;
  randomize_cc2 = kDefStepRandomizeCC2;
  retrigger = kDefStepRetrigger;
  retrigger_velocity = kDefStepRetriggerVelocity;
  retrigger_transposition = kDefStepRetriggerTransposition;
  retrigger_transposition_scale = kDefStepRetriggerTranspositionScale;
  retrigger_condition = kDefStepRetriggerCondition;

  memset(reserved, 0, sizeof(reserved));
}

#define FIX_DEF(name, Name) \
  if (name < kMin##Name || name > kMax##Name) { \
    name = kDef##Name; \
  }

void Seq::Step::Validate() {
  FIX_DEF(note, StepNote);
  FIX_DEF(velocity, StepVelocity);
  FIX_DEF(gate, StepGate);
  FIX_DEF(cc1, StepCC);
  FIX_DEF(cc2, StepCC);
  FIX_DEF(condition, StepCondition);
  FIX_DEF(cc1_condition, StepCC1Condition);
  FIX_DEF(cc2_condition, StepCC2Condition);
  FIX_DEF(randomize_note, StepRandomizeNote);
  FIX_DEF(randomize_note_scale, StepRandomizeNoteScale);
  FIX_DEF(randomize_velocity, StepRandomizeVelocity);
  FIX_DEF(randomize_cc1, StepRandomizeCC1);
  FIX_DEF(randomize_cc2, StepRandomizeCC2);
  FIX_DEF(retrigger, StepRetrigger);
  FIX_DEF(retrigger_velocity, StepRetriggerVelocity);
  FIX_DEF(retrigger_transposition, StepRetriggerTransposition);
  FIX_DEF(retrigger_transposition_scale, StepRetriggerTranspositionScale);
  FIX_DEF(retrigger_condition, StepRetriggerCondition);
}

#undef FIX_DEF

void Seq::Init() {
  step_[0].Init();
  for (uint8_t n = 1; n < kNumSteps; n++) {
    step_[n] = step_[0];
  }
}

void Seq::InitStep(uint8_t step) {
  step_[step].Init();
}

void Seq::Validate() {
  for (uint8_t n = 0; n < kNumSteps; n++) {
    step_[n].Validate();
  }
}

uint8_t Seq::all_steps_skipped() const {
  for (uint8_t n = 0; n < kNumSteps; n++) {
    if (!(step_[n].flags & kSkip))
      return 0;
  }
  return 1;
}

uint8_t Seq::non_skipped_steps_count() const {
  uint8_t count = 0;
  for (uint8_t n = 0; n < kNumSteps; n++) {
    if (!(step_[n].flags & kSkip)) {
      ++count;
    }
  }
  return count;
}

void Seq::get_steps(Step steps[kNumSteps]) const {
  for (uint8_t n = 0; n < kNumSteps; n++) {
    steps[n] = step_[n];
  }
}

void Seq::set_steps(const Step steps[kNumSteps]) {
  for (uint8_t n = 0; n < kNumSteps; n++) {
    step_[n] = steps[n];
  }
}

const Seq::Step& Seq::get_step(uint8_t step) const {
  return step_[step];
}

void Seq::set_step(uint8_t step, const Step& stepData) {
  step_[step] = stepData;
}

/* static */
void Seq::ReverseSteps(Step* pStep, uint8_t left, uint8_t right) {
  Step* pStep1 = pStep + left;
  Step* pStep2 = pStep + right;
  while (pStep1 < pStep2) {
    Step step = *pStep1;
    *pStep1 = *pStep2;
    *pStep2 = step;
    pStep1++;
    pStep2--;
  }
}

}  // midialf
