// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Sequence model.

#ifndef MIDIALF_SEQ_H_
#define MIDIALF_SEQ_H_

#include "midialf/midialf.h"
#include "midialf/duration.h"
#include "midialf/cond.h"

namespace midialf {
  
class Seq {
public:

  Seq() {}

  void Init();
  void InitStep(uint8_t step);

  void Validate();

  struct Step {
    Step() {}

    void Init();
    void Validate();

    uint8_t note;
    uint8_t velocity;
    uint8_t gate;
    uint8_t cc1;
    uint8_t cc2;
    uint16_t flags;
    uint8_t condition;
    uint8_t cc1_condition;
    uint8_t cc2_condition;
    uint8_t randomize_note;
    uint8_t randomize_note_scale;
    uint8_t randomize_velocity;
    uint8_t randomize_cc1;
    uint8_t randomize_cc2;
    uint8_t retrigger;
    uint8_t retrigger_velocity;
    uint8_t retrigger_transposition;
    uint8_t retrigger_transposition_scale;
    uint8_t retrigger_condition;
    uint8_t reserved[9];
  };

  enum StepFlags {
    kMute    = 0x01,
    kSkip    = 0x02,
    kLegato  = 0x04,
    kSendCC1 = 0x10,
    kSendCC2 = 0x20,
    kHasCondition          = 0x0100,
    kHasCC1Condition       = 0x0200,
    kHasCC2Condition       = 0x0400,
    kHasRetrigger          = 0x0800,
    kHasRetriggerCondition = 0x1000,
  };

  // Accessors

  #define DEFINE_STEP_ACCESSORS(type, name) \
    type step_##name(uint8_t step) const { return step_[step].name; } \
    void set_step_##name(uint8_t step, type name) { step_[step].name = name; } 

  #define DEFINE_STEP_FLAG_ACCESSORS(name, flag) \
    uint8_t step_##name(uint8_t step) const { return (step_[step].flags & flag) ? 1 : 0; } \
    void set_step_##name(uint8_t step, uint8_t name) { SETFLAGTO(step_[step].flags, flag, name); } 

  DEFINE_STEP_ACCESSORS(uint8_t, note)
  DEFINE_STEP_ACCESSORS(uint8_t, velocity)
  DEFINE_STEP_ACCESSORS(uint8_t, gate)
  DEFINE_STEP_ACCESSORS(uint8_t, cc1)
  DEFINE_STEP_ACCESSORS(uint8_t, cc2)
  DEFINE_STEP_ACCESSORS(uint16_t, flags)
  DEFINE_STEP_ACCESSORS(uint8_t, condition)
  DEFINE_STEP_ACCESSORS(uint8_t, cc1_condition)
  DEFINE_STEP_ACCESSORS(uint8_t, cc2_condition)
  DEFINE_STEP_ACCESSORS(uint8_t, randomize_note)
  DEFINE_STEP_ACCESSORS(uint8_t, randomize_note_scale)
  DEFINE_STEP_ACCESSORS(uint8_t, randomize_velocity)
  DEFINE_STEP_ACCESSORS(uint8_t, randomize_cc1)
  DEFINE_STEP_ACCESSORS(uint8_t, randomize_cc2)
  DEFINE_STEP_ACCESSORS(uint8_t, retrigger)
  DEFINE_STEP_ACCESSORS(uint8_t, retrigger_velocity)
  DEFINE_STEP_ACCESSORS(uint8_t, retrigger_transposition)
  DEFINE_STEP_ACCESSORS(uint8_t, retrigger_transposition_scale)
  DEFINE_STEP_ACCESSORS(uint8_t, retrigger_condition)

  DEFINE_STEP_FLAG_ACCESSORS(mute, kMute)
  DEFINE_STEP_FLAG_ACCESSORS(skip, kSkip)
  DEFINE_STEP_FLAG_ACCESSORS(legato, kLegato)
  DEFINE_STEP_FLAG_ACCESSORS(send_cc1, kSendCC1)
  DEFINE_STEP_FLAG_ACCESSORS(send_cc2, kSendCC2)
  DEFINE_STEP_FLAG_ACCESSORS(has_condition, kHasCondition)
  DEFINE_STEP_FLAG_ACCESSORS(has_cc1_condition, kHasCC1Condition)
  DEFINE_STEP_FLAG_ACCESSORS(has_cc2_condition, kHasCC2Condition)
  DEFINE_STEP_FLAG_ACCESSORS(has_retrigger, kHasRetrigger)
  DEFINE_STEP_FLAG_ACCESSORS(has_retrigger_condition, kHasRetriggerCondition)

  #undef DEFINE_STEP_ACCESSORS
  #undef DEFINE_FLAG_ACCESSORS

  uint8_t all_steps_skipped() const;
  uint8_t non_skipped_steps_count() const;

  void get_steps(Step steps[kNumSteps]) const;
  void set_steps(const Step steps[kNumSteps]);

  const Step& get_step(uint8_t step) const;
  void set_step(uint8_t step, const Step& stepData);

  Step* steps_ptr() { return &step_[0]; }

  static void ReverseSteps(Step* pStep, uint8_t left, uint8_t right);

private:
  Step step_[kNumSteps];

};

}  // namespace midialf

#endif // MIDIALF_SEQ_H_
