// Copyright 2015 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Stored slot implementation.

#include "midialf/slot.h"
#include "midialf/eemem.h"
#include "midialf/storage.h"

namespace midialf {

using namespace avrlib;

void Slot::Init() {
  size_ = sizeof(SlotData);
  memset(name_, '_', kNameLength);
  channel_ = kDefChannel;
  tempo_ = kDefTempo;
  step_length_ = kDefStepLength;
  swing_ = kDefSwing;
  seq_root_note_ = kDefSeqRootNote;
  seq_link_mode_ = kDefSeqLinkMode;
  seq_switch_mode_ = kDefSeqSwitchMode;
  seq_transpose_mode_ = kDefSeqTransposeMode;
  seq_direction_ = kDefSeqDirection;
  steps_forward_ = kDefStepCount;
  steps_backward_ = kDefStepCount;
  steps_replay_ = kDefStepCount;
  steps_interval_ = kDefStepCount;
  steps_repeat_ = kDefStepCount;
  steps_skip_ = kDefStepCount;
  cc1_number_ = kDefCC1Number;
  cc2_number_ = kDefCC2Number;
  prog_change_mode_ = kDefProgChangeMode;
  ctrl_change_mode_ = kDefCtrlChangeMode;
  clock_mode_ = kDefClockMode;
  bank_select_msb_ = kDefBankSelectMsb;
  bank_select_lsb_ = kDefBankSelectLsb;
  program_change_ = kDefProgramChange;
  flags_ = kDefFlags;

  cv_mode_[0] = kDefCV1Mode;
  cv_mode_[1] = kDefCV2Mode;
  cv_mode_[2] = kDefCV3Mode;
  cv_mode_[3] = kDefCV4Mode;
  gate_mode_[0] = kDefGate1Mode;
  gate_mode_[1] = kDefGate2Mode;
  gate_mode_[2] = kDefGate3Mode;
  gate_mode_[3] = kDefGate4Mode;
  strobe_width_ = kDefStrobeWidth;

  memset(reserved_, 0, sizeof(reserved_));

  for (uint8_t n = 0; n < kNumSeqs; n++) {
    seq_[n].Init();
  }

  song_.Init();

  for (uint8_t n = 0; n < kNumLfos; n++) {
    lfo_[n].Init(n);
  }

  lfo_resolution_ = kDefLfoResolution;
}

void Slot::Read(uint8_t slot) {
  Init();

  uint16_t address = storage.slot_address(slot);
  if (address == Storage::kBadAddress)
    return;

  uint16_t size;
  if (storage.Read(&size, address, sizeof(size)) != sizeof(size))
    return;

  if (size > sizeof(SlotData))
    return;

  if (storage.Read(this, address, size) != size) {
    Init();
    return;
  }

  Validate();
}

/* static */
void Slot::ReadName(uint8_t slot, uint8_t name[kNameLength]) {
  memset(name, '_', kNameLength);

  uint16_t address = storage.slot_address(slot);
  if (address == Storage::kBadAddress)
    return;

  uint8_t size;
  if (storage.Read(&size, address, sizeof(size)) != sizeof(size))
    return;

  if (size > sizeof(SlotData))
    return;

  if (storage.Read(name, address + OFFSETOF(SlotData, name_), kNameLength) != kNameLength) {
    memset(name, '_', kNameLength);
    return;
  }

  ValidateName(name, kNameLength);
}

void Slot::Write(uint8_t slot) {
  uint16_t address = storage.slot_address(slot);
  if (address == Storage::kBadAddress)
    return;

  storage.Write(this, address, sizeof(SlotData));
}

#define FIX_DEF_(name, Name) \
  if (name < kMin##Name || name > kMax##Name) { \
    name = kDef##Name; \
  }

#define FIX_DEF2(name, Name, Def) \
  if (name < kMin##Name || name > kMax##Name) { \
    name = Def; \
  }

void Slot::Validate() {
  ValidateName(name_, kNameLength);

  if (channel_ > 0xf) {
Fix:Init();
    return;
  }

  FIX_DEF_(tempo_, Tempo)
  FIX_DEF_(step_length_, StepLength)
  FIX_DEF_(swing_, Swing)
  FIX_DEF_(seq_root_note_, SeqRootNote)
  FIX_DEF_(seq_link_mode_, SeqLinkMode)
  FIX_DEF_(seq_switch_mode_, SeqSwitchMode)
  FIX_DEF_(seq_transpose_mode_, SeqTransposeMode)
  FIX_DEF_(seq_direction_, SeqDirection)
  FIX_DEF_(steps_forward_, StepCount)
  FIX_DEF_(steps_backward_, StepCount)
  FIX_DEF_(steps_replay_, StepCount)
  FIX_DEF_(steps_interval_, StepCount)
  FIX_DEF_(steps_repeat_, StepCount)
  FIX_DEF_(steps_skip_, StepCount)
  FIX_DEF2(cc1_number_, CCNumber, kDefCC1Number)
  FIX_DEF2(cc2_number_, CCNumber, kDefCC2Number)
  FIX_DEF_(prog_change_mode_, ProgChangeMode)
  FIX_DEF_(ctrl_change_mode_, CtrlChangeMode)
  FIX_DEF_(clock_mode_, ClockMode); if (clock_mode_ & CLOCK_MODE_CONTIGUOUS) clock_mode_|= CLOCK_MODE_SENDOUTPUT;
  FIX_DEF_(bank_select_msb_, BankSelectMsb)
  FIX_DEF_(bank_select_lsb_, BankSelectLsb)

  for (uint8_t n = 0; n < numbof(cv_mode_); n++) {
    FIX_DEF_(cv_mode_[n], CVMode)
  }

  for (uint8_t n = 0; n < numbof(gate_mode_); n++) {
    FIX_DEF_(gate_mode_[n], GateMode)
  }

  FIX_DEF_(strobe_width_, StrobeWidth)

  for (uint8_t n = 0; n < kNumSeqs; n++) {
    seq_[n].Validate();
  }

  song_.Validate();

  for (uint8_t n = 0; n < kNumLfos; n++) {
    lfo_[n].Validate();
  }

  FIX_DEF_(lfo_resolution_, LfoResolution)
}

/* static */
void Slot::ValidateName(uint8_t name[], uint8_t length) {
  for (uint8_t n = 0; n < length; n++) {
    if (name[n] < kMinNameChar || name[n] > kMaxNameChar) {
      name[n] = '_';
    }
  }
}

}  // namespace midialf

