// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Sysex routines.

#include "midialf/sysex.h"
#include "midialf/storage.h"
#include "midialf/ui.h"

#include <stdlib.h>
#include <avr/eeprom.h>

namespace midialf {

using namespace avrlib;

/* extern */
Sysex sysex;

static const uint8_t kSysexSpacingDelay = 100;

static const prog_uint8_t header[] PROGMEM = {
  0xf0,  // <Sysex>
  0x29,  // PPG manufacturer ID
  'A','L','F',0x01, // ALF Product ID
  // * Command byte (SysexCmd)
  // * Argument byte (command modifier)
  // * Payload bytes (nibbelized)
  // 0xf7 EOX byte
};

//<static>
uint16_t Sysex::bytes_sent_;
uint16_t Sysex::bytes_received_;
uint16_t Sysex::buffer_size_;
uint8_t* Sysex::buffer_;
uint8_t Sysex::state_;
uint8_t Sysex::error_code_;
uint8_t Sysex::prev_state_;
uint8_t Sysex::command_[2];
uint8_t Sysex::cancel_;
uint8_t Sysex::slot_;
//</static>

///////////////////////////////////////////////////////////////////////////////
// Sysex send routines

/* static */
void Sysex::SendSlotData(uint8_t slot_numb) {
  if (cancel_) {
    state_ = SENDING_SYSEX_CANCELED;
    return;
  }

  slot_ = slot_numb;
  state_ = SENDING_SYSEX;
  ui.request_redraw(); ui.DoEvents(); ui.RedrawScreen();

  Slot slot; dev.ToSlot(slot);
  SendSysexHeader(SYSEXCMD_SLOTDATA, slot_numb);
  SendByteData(&slot, sizeof(slot));
  SendSysexEnd();
}

/* static */
void Sysex::SendSlot(uint8_t slot_numb) {
  cancel_ = 0;
  bytes_sent_ = 0;
  uint8_t cur_slot = dev.slot();
  dev.LoadSlot(slot_numb);
  SendSlotData(kCurSlot);
  dev.LoadSlot(cur_slot);
  state_ = SENDING_SYSEX_DONE;
}

/* static */
void Sysex::SendCurSlot() {
  cancel_ = 0;
  bytes_sent_ = 0;
  SendSlotData(kCurSlot);
  state_ = SENDING_SYSEX_DONE;
}

/* static */
void Sysex::SendAllSlots() {
  cancel_ = 0;
  bytes_sent_ = 0;
  uint8_t cur_slot = dev.slot();
  uint8_t num_slots = min(storage.num_slots(), kNumSlots);
  for (uint8_t n = 0; n < num_slots && !cancel_; n++) {
    dev.LoadSlot(n);
    SendSlotData(n);
  }
  dev.LoadSlot(cur_slot);
  state_ = SENDING_SYSEX_DONE;
}

/* static */
void Sysex::SendTune() {
#ifdef ENABLE_CV_EXT
  cancel_ = 0;
  bytes_sent_ = 0;

  slot_ = kCurSlot;
  state_ = SENDING_SYSEX;
  ui.request_redraw(); ui.DoEvents(); ui.RedrawScreen();

  int16_t* ptr = cv.GetTuneAddr();
  SendSysexHeader(SYSEXCMD_TUNEDATA, 0);
  SendByteData(ptr, sizeof(int16_t) * 128);
  SendSysexEnd();

  state_ = SENDING_SYSEX_DONE;
#endif // #ifdef ENABLE_CV_EXT
}

/* static */
void Sysex::SendSysexHeader(uint8_t cmd, uint8_t arg) {
  // Send sysex header
  for (uint8_t n = 0; n < sizeof(header); n++) {
    SendByte(pgm_read_byte(header + n));
  }

  // Send sysex cmd code and arg
  SendByte(cmd);
  SendByte(arg);
}

/* static */
void Sysex::SendByteData(const void* data, uint16_t size) {
  const uint8_t* ptr = static_cast<const uint8_t*>(data);
  for (uint16_t n = 0; n < size; ++n, ++ptr) {
    SendByteData(*ptr);
  }
}

/* static */
void Sysex::SendByteData(uint8_t byte) {
  SendByte(U8ShiftRight4(byte));
  SendByte(byte & 0x0f);
}

/* static */
void Sysex::SendSysexEnd() {
  SendByte(0xf7);
  _delay_ms(kSysexSpacingDelay);
}

/* static */
void Sysex::SendByte(uint8_t byte) {
  midi_out.Send(byte);
  ++bytes_sent_;
}

///////////////////////////////////////////////////////////////////////////////
// Sysex receive routines

/* static */
void Sysex::Receive(uint8_t byte) {
  if (byte == 0xf0) {
    bytes_received_ = 0;
    state_ = RECEIVING_HEADER;
  }
  switch (state_) {
    case RECEIVING_HEADER:
      if (pgm_read_byte(header + bytes_received_) == byte) {
        bytes_received_++;
        if (bytes_received_ >= sizeof(header)) {
          state_ = RECEIVING_COMMAND;
          bytes_received_ = 0;
        }
      } else {
        state_ = RECEPTION_ALIEN;
      }
      break;

    case RECEIVING_COMMAND:
      command_[bytes_received_++] = byte;
      if (bytes_received_ == 2) {
        switch (command_[0]) {
          case SYSEXCMD_SLOTDATA: 
          case SYSEXCMD_REQSLOTDATA:
            slot_ = command_[1];
            break;
          case SYSEXCMD_TUNEDATA:
          case SYSEXCMD_REQTUNEDATA:
            slot_ = kCurSlot;
            break;
        }

        bytes_received_ = 0;
        AllocateBuffer();
        if (!buffer_) {
          state_ = RECEPTION_ERROR;
          error_code_ = sysexReceiveError_noBufferMemory;
        } else {
          state_ = RECEIVING_DATA;
          error_code_ = noSysexReceiveError;
        }
        ui.RequestShowPage(PAGE_SYSEX);
      }
      break;

    case RECEIVING_DATA:
      if (!buffer_) {
        state_ = RECEPTION_ERROR;
        error_code_ = sysexReceiveError_noBufferMemory;
        ui.request_redraw();
      } else
      if (byte != 0xf7) {
        uint16_t n = bytes_received_ >> 1;
        if (n >= buffer_size_) {
          state_ = RECEPTION_ERROR;
          error_code_ = sysexReceiveError_exceededBufferMemory;
          ui.request_redraw();
        } else {
          if (++bytes_received_ & 1) {
            buffer_[n] = U8ShiftLeft4(byte);
          } else {
            buffer_[n]|= byte & 0xf;
          }
        }
      } else {
        if (bytes_received_ & 1) {
          state_ = RECEPTION_ERROR;
          error_code_ = sysexReceiveError_unevenBytesReceived;
        } else {
          state_ = RECEPTION_OK;
          ProcessSysex();
        }
        // If EOX is not received, the buffer will be released when
        // the user leaves Sysex UI page.
        FreeBuffer();
        ui.request_redraw();
      }
      break;
  }
}

/* static */
void Sysex::ProcessSysex() {
  switch (command_[0]) {
    case SYSEXCMD_SLOTDATA:
      if (bytes_received_ >> 1 != sizeof(Slot)) {
        state_ = RECEPTION_ERROR;
        error_code_ = sysexReceiveError_invalidBytesReceived;
        return;
      }

      ((Slot*)buffer_)->Validate();

      dev.FromSlot(*((Slot*)buffer_));

      if (command_[1] < min(storage.num_slots(), kNumSlots)) {
        dev.SaveSlot(command_[1]);
      }
      return;

#ifdef ENABLE_CV_EXT
    case SYSEXCMD_TUNEDATA:
      if (bytes_received_ >> 1 != kCvTuneSize) {
        state_ = RECEPTION_ERROR;
        error_code_ = sysexReceiveError_invalidBytesReceived;
        return;
      }

      memcpy(cv.GetTuneAddr(), buffer_, kCvTuneSize);

      return;
#endif

    case SYSEXCMD_REQSLOTDATA:
      ui.AddRequest(REQUEST_SLOTDATA, command_[1]);
      return;

#ifdef ENABLE_CV_EXT
    case SYSEXCMD_REQTUNEDATA:
      ui.AddRequest(REQUEST_TUNEDATA, command_[1]);
      return;
#endif

    default:
      state_ = RECEPTION_ERROR;
      error_code_ = sysexReceiveError_unexpectedSysexCmd;
      return;
  }
}

/* static */
void Sysex::AllocateBuffer() {
  uint16_t cb;
  switch (command_[0]) {
    case SYSEXCMD_SLOTDATA: cb = sizeof(SlotData); break;
#ifdef ENABLE_CV_EXT
    case SYSEXCMD_TUNEDATA: cb = kCvTuneSize; break;
#endif
    case SYSEXCMD_REQSLOTDATA:
    case SYSEXCMD_REQTUNEDATA: // fall through
    default: FreeBuffer(); return;
  }

  if (buffer_) {
    if (buffer_size_ == cb)
      return;

    free(buffer_);
  }

  buffer_ = (uint8_t*)malloc(cb);
  buffer_size_ = cb;
}

/* static */
void Sysex::FreeBuffer() {
  if (buffer_) { 
    free(buffer_); 
    buffer_ = 0; 
    buffer_size_ = 0;
  }
}

/* static */
void Sysex::Forward(uint8_t byte) {
  if (state_ == RECEPTION_ALIEN) {
    if (prev_state_ == RECEIVING_HEADER) {
      for (uint8_t n = 0; n < bytes_received_; n++) {
        SendByte(pgm_read_byte(header + n));
      }
    }
    SendByte(byte);
  }

  prev_state_ = state_;
}

}  // namespace midialf
