// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Sysex routines.

#ifndef MIDIALF_SYSEX_H_
#define MIDIALF_SYSEX_H_

#include "midialf/midialf.h"
#include "midialf/slot.h"
#include "midialf/dev.h"
#ifdef ENABLE_CV_EXT
#include "midialf/cv/cv.h"
#endif

namespace midialf {

enum SysexCmd {
  SYSEXCMD_SLOTDATA    = 0x01, // slot data, argument specifies slot index (0x7f for current one)
  SYSEXCMD_TUNEDATA    = 0x02, // 128 x int16 tune data points
  SYSEXCMD_REQSLOTDATA = 0x11, // request slot data, argument specifies slot index (0x7f for current one)
  SYSEXCMD_REQTUNEDATA = 0x12, // 128 x int16 tune data points
};

enum SysexState {
  SENDING_SYSEX,
  SENDING_SYSEX_DONE,
  SENDING_SYSEX_CANCELED,
  RECEIVING_HEADER,
  RECEIVING_COMMAND,
  RECEIVING_DATA,
  RECEPTION_OK,
  RECEPTION_ERROR,
  RECEPTION_ALIEN,
};

enum SysexReceiveErrorCodes {
  noSysexReceiveError,
  sysexReceiveError_noBufferMemory,       // 1
  sysexReceiveError_exceededBufferMemory, // 2
  sysexReceiveError_unevenBytesReceived,  // 3
  sysexReceiveError_invalidBytesReceived, // 4
  sysexReceiveError_unexpectedSysexCmd,   // 5
};

static const uint8_t kCurSlot = 0x7f;

class Sysex {
 public:
  Sysex() {}

  static void SendSlot(uint8_t slot_numb);
  static void SendCurSlot();
  static void SendAllSlots();
  static void SendTune();
  
  static void Receive(uint8_t byte);
  static void Forward(uint8_t byte);

  static uint8_t state() { return state_; }
  static uint8_t error_code() { return error_code_; }
  static uint16_t bytes_sent() { return bytes_sent_; }
  static uint16_t bytes_received() { return bytes_received_; }
  static uint8_t slot() { return slot_; }
  
  static void AllocateBuffer();
  static void FreeBuffer();

  static void Cancel() {
    cancel_ = 1;
    FreeBuffer();
  }

  static uint8_t IsSendingSysex() {
    return state_ == SENDING_SYSEX;
  }

  static uint8_t IsReceivingSysex() {
    return state_ >= RECEIVING_HEADER && state_ <= RECEIVING_DATA;
  }

private:

  static void SendSlotData(uint8_t slot_numb);

  static void SendSysexHeader(uint8_t cmd, uint8_t arg = 0);
  static void SendByteData(const void* data, uint16_t size);
  static void SendByteData(uint8_t byte);
  static void SendByte(uint8_t byte);

  static void SendSysexEnd();

  static void ProcessSysex();
  
  static uint16_t bytes_sent_;
  static uint16_t bytes_received_;
  static uint16_t buffer_size_;
  static uint8_t* buffer_;
  static uint8_t state_;
  static uint8_t error_code_;
  static uint8_t prev_state_;
  static uint8_t command_[2];
  static uint8_t cancel_;
  static uint8_t slot_;
  
  DISALLOW_COPY_AND_ASSIGN(Sysex);
};

extern Sysex sysex;

}  // namespace midialf

#endif // MIDIALF_SYSEX_H_
