// Copyright 2012 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// Based on Ambika code by Olivier Gillet (ol.gillet@gmail.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------

#include "midialf/ui.h"

#include "avrlib/op.h"
#include "avrlib/string.h"

#include "midialf/leds.h"
#include "midialf/clock.h"
#include "midialf/display.h"
#include "midialf/duration.h"
#include "midialf/sysex.h"
#include "midialf/slot.h"
#include "midialf/dev.h"

#include "midialf/ui_pages/note_page.h"
#include "midialf/ui_pages/velocity_page.h"
#include "midialf/ui_pages/gate_page.h"
#include "midialf/ui_pages/cc_page.h"
#include "midialf/ui_pages/config_page.h"
#include "midialf/ui_pages/lfo_page.h"
#include "midialf/ui_pages/song_page.h"
#include "midialf/ui_pages/dlg_page.h"
#include "midialf/ui_pages/step_page.h"
#include "midialf/ui_pages/load_page.h"
#include "midialf/ui_pages/save_page.h"
#include "midialf/ui_pages/sysex_page.h"
#include "midialf/ui_pages/command_page.h"
#include "midialf/ui_pages/tune_page.h"

namespace midialf {

using namespace avrlib;

static const uint8_t kDblClickTime = 250;

/* <static> */
EncoderA Ui::encoderA_;
EncoderB Ui::encoderB_;
Encoders Ui::encoders_;

StepSwitches Ui::stepSwitches_;
SideSwitches Ui::sideSwitches_;

EventQueue<32> Ui::queue_;

UiPageIndex Ui::active_page_;
UiPageIndex Ui::last_page_;
EventHandlers Ui::event_handlers_;
PageInfo Ui::page_info_;

uint8_t Ui::cycle_;
uint8_t Ui::request_redraw_;
uint8_t Ui::inhibit_shift_raised_;

uint8_t Ui::last_click_encoder_;
uint32_t Ui::last_click_time_;

uint8_t Ui::step_blink_count_;
uint8_t Ui::tick_count_;

/* </static> */

static RunningCallback prev_running_callback;
static TickCallback prev_tick_callback;
static StepCallback prev_step_callback;

/* extern */
Ui ui;

const prog_PageInfo page_registry[] PROGMEM = {
  { PAGE_NOTE,
    &NotePage::event_handlers_,
  },
  { PAGE_VELOCITY,
    &VelocityPage::event_handlers_,
  },
  { PAGE_GATE,
    &GatePage::event_handlers_,
  },
  { PAGE_CC,
    &CCPage::event_handlers_,
  },
  { PAGE_CONFIG,
    &ConfigPage::event_handlers_,
  },
  { PAGE_LFO,
    &LfoPage::event_handlers_,
  },
  { PAGE_SONG,
    &SongPage::event_handlers_,
  },
  { PAGE_DLG,
    &DlgPage::event_handlers_,
  },
  { PAGE_STEP,
    &StepPage::event_handlers_,
  },
  { PAGE_LOAD,
    &LoadPage::event_handlers_,
  },
  { PAGE_SAVE,
    &SavePage::event_handlers_,
  },
  { PAGE_SYSEX,
    &SysexPage::event_handlers_,
  },
  { PAGE_COMMAND,
    &CommandPage::event_handlers_,
  },
#ifdef ENABLE_CV_EXT
  { PAGE_TUNE,
    &TunePage::event_handlers_,
  },
#endif
};

/* static */
void Ui::Init() {
  encoderA_.Init();
  encoderB_.Init();
  encoders_.Init();
  stepSwitches_.Init();
  sideSwitches_.Init();
  lcd.Init();
  display.Init();

  lcd.SetCustomCharMapRes(chr_res_custom_chars, kNumCustomChars, kCustomChars);

  // Init device if ENCB is down
  encoderB_.Read();
  if (encoderB_.immediate_value() == 0) {
    InitDev();
  }

  Logo();

  ShowPage(FIRST_PAGE);

  prev_running_callback = dev.set_running_callback(RunningCallback);
  prev_step_callback = dev.set_step_callback(StepCallback);
  prev_tick_callback = dev.set_tick_callback(TickCallback);
}

/* static */
void Ui::InitDev() {
  ClearScreen();
  char* line1 = display.line_buffer(0);
  memcpy_P(&line1[0], PSTRN("Initializing device..."));
  RedrawScreen();
  dev.InitDev();
}

/* static */
void Ui::Logo() {
  char* line1 = display.line_buffer(0);
#ifdef ENABLE_CV_EXT
  static const prog_char logo[] PROGMEM = "MidiALF/CV v" MIDIALF_VERSION;
#else
  static const prog_char logo[] PROGMEM = "MidiALF v" MIDIALF_VERSION;
#endif
  memcpy_P(&line1[0], logo, lengof(logo));

  uint8_t x = lengof(logo) + 2;
  line1[x++] = '(';
  x+= UnsafeItoaLen(sizeof(Seq), 3, &line1[x]);
  line1[x++] = '/';
  x+= UnsafeItoaLen(sizeof(SlotData), 4, &line1[x]);
  line1[x++] = ')';

  RedrawScreen();

  // Run led check sequence

  static const prog_uint8_t xleds[] PROGMEM = {
    LED_5     | (LED_4    << 4),
    LED_6     | (LED_3    << 4),
    LED_7     | (LED_2    << 4),
    LED_8     | (LED_1    << 4),
    LED_RUN   | (LED_SEQA << 4),
    LED_SAVE  | (LED_SEQB << 4),
    LED_LOAD  | (LED_SEQC << 4),
    LED_SHIFT | (LED_SEQD << 4),
    LED_LOAD  | (LED_SEQC << 4),
    LED_SAVE  | (LED_SEQB << 4),
    LED_RUN   | (LED_SEQA << 4),
    LED_8     | (LED_1    << 4),
    LED_7     | (LED_2    << 4),
    LED_6     | (LED_3    << 4),
    LED_5     | (LED_4    << 4),
  };

  for (uint8_t n = 0; n < 1; n++) {
    for (uint8_t i = 0; i < numbof(xleds); i++) {
      uint8_t byte = pgm_read_byte(&xleds[i]);
      uint8_t pixel1 = byte & 0xf;
      uint8_t pixel2 = U8ShiftRight4(byte);
      leds.set_pixel(pixel1);
      leds.set_pixel(pixel2);
      leds.Write(); _delay_ms(50);
      leds.clr_pixel(pixel1);
      leds.clr_pixel(pixel2);
    }
  }

  _delay_ms(250);
  leds.Clear();
  leds.Write();
}

/* static */
void Ui::Poll() {

  // 2.45KHz
  cycle_++;

  // Poll Encoder A
  { int8_t increment = encoderA_.Read();
    if (increment) {
      if (IsShifted()) increment*= 10;
      queue_.AddEvent(CONTROL_ENCODER, ENCODER_A, increment);
    }
    if (encoderA_.clicked()) {    
      queue_.AddEvent(CONTROL_ENCODER_CLICK, ENCODER_A, GetEncoderClickValue(ENCODER_A));
    }
  }

  // Poll Encoder B
  { int8_t increment = encoderB_.Read();
    if (increment) {
      if (IsShifted()) increment*= 10;
      queue_.AddEvent(CONTROL_ENCODER, ENCODER_B, increment);
    }

    if (encoderB_.clicked()) {    
      queue_.AddEvent(CONTROL_ENCODER_CLICK, ENCODER_B, GetEncoderClickValue(ENCODER_B));
    }
  }

  // Poll step encoders
  { encoders_.Poll();
    for (uint8_t i = 0; i < 8; ++i) {
      int8_t increment = encoders_.Read(i);
      if (increment) {
        if (IsShifted()) increment*= 10;
        queue_.AddEvent(CONTROL_ENCODER, ENCODER_1 + i, increment);
      }
      if (encoders_.lowered(i)) {
        uint8_t id = ENCODER_1 + i;
        queue_.AddEvent(CONTROL_ENCODER_CLICK, id, GetEncoderClickValue(id));
      }
    }
  }

  // Poll switches
  if ((cycle_ & 3) == 0) {
    // 612Hz
    stepSwitches_.Read();
    sideSwitches_.Read();
    for (uint8_t i = 0; i < 8; ++i) {
      if (stepSwitches_.lowered(i)) {
        queue_.AddEvent(CONTROL_SWITCH, SWITCH + IsShifted(), i);
      }
      if (sideSwitches_.lowered(i)) {
        if (i != SIDE_SWITCH_SHIFT) {
          queue_.AddEvent(CONTROL_SWITCH, SIDE_SWITCH + IsShifted(), i);
        } else
          queue_.AddEvent(CONTROL_SWITCH, SIDE_SWITCH, i);
      } else
      if (sideSwitches_.raised(i)) {
        if (inhibit_shift_raised_ && i == SIDE_SWITCH_SHIFT) {
          inhibit_shift_raised_ = 0;
        } else
          queue_.AddEvent(CONTROL_SWITCH, SIDE_SWITCH_RAISED, i);
      }
    }
  }

  // Output
  lcd.Tick();
}

/* static */
void Ui::DoEvents() {
  display.Tick();
  
  uint8_t redraw = request_redraw_; request_redraw_ = 0;
  while (queue_.available()) {
    Event e = queue_.PullEvent();
    queue_.Touch();
    HandleEvent(e);
    redraw = 1;
  }
  
  if (queue_.idle_time_ms() > 1000) {
    queue_.Touch();
    if ((*event_handlers_.OnIdle)()) {
      redraw = 1;
    }
  }
  
  if (redraw) {
    ClearScreen();
    if (!(*event_handlers_.UpdateScreen)()) {
      HandleUpdateScreen();
    }
  }
  
  if (!(*event_handlers_.UpdateLeds)()) {
    HandleUpdateLeftSideLeds();
    HandleUpdateRightSideLeds();
    HandleUpdateStepLeds();
  }
}

/* static */
void Ui::HandleEvent(const Event& e) {
  switch (e.control_type) {
    case CONTROL_ENCODER:
      if (!(*event_handlers_.OnIncrement)(e.control_id, e.value)) {
        HandleEncoderEvent(e);
      }
      break;
    case CONTROL_ENCODER_CLICK:
      if (!(*event_handlers_.OnClick)(e.control_id, e.value)) {
        HandleEncoderClickEvent(e);
      }
      break;
    case CONTROL_SWITCH:
      if (!(*event_handlers_.OnSwitch)(e.control_id, e.value)) {
        HandleSwitchEvent(e);
      }
      break;
    case CONTROL_REQUEST:
      {
        HandleRequestEvent(e);
      }
      break;
  }
}

/* static */
void Ui::HandleEncoderEvent(const Event& e) {
  // ENCA does nothing by default
  if (e.control_id == ENCODER_A) {
    ;
  } else
  // ENCB cycles between primary pages
  if (e.control_id == ENCODER_B) {
    uint8_t page = active_page_;
    if ((int8_t)e.value > 0) {
      if (page < LAST_PAGE) {
        page++;
      }
    } else
    if ((int8_t)e.value < 0) {
      if (page > FIRST_PAGE) {
        page--;
      }
    }
    ShowPage(static_cast<UiPageIndex>(page), /* remember_last_page */ 0);
  }
}

/* static */
void Ui::HandleEncoderClickEvent(const Event& e) {
  if (e.value == CLICK) {
    if (e.control_id == ENCODER_A) {
      ShowPage(PAGE_COMMAND);
    } else
    if (e.control_id == ENCODER_B) {
      ShowPage(PAGE_CONFIG);
    } else
    // Show step configuration page on step encoder click or select step if recording
    if (e.control_id >= ENCODER_1 && e.control_id <= ENCODER_8) {
      uint8_t step = e.control_id - ENCODER_1;
      if (dev.recording()) {
        dev.set_step(step);
        dev.SendViewSeqStep(step);
      } else
        ShowStepPage(step);
    }
  } else

  if (e.value == CLICK_SHIFTED) {
    if (e.control_id == ENCODER_A) {
      ShowPage(FIRST_PAGE);
    } else
    if (e.control_id == ENCODER_B) {
      dev.SilenceAllNotes();
    } else
    // Shifr click manually selects step
    if (e.control_id >= ENCODER_1 && e.control_id <= ENCODER_8) {
      uint8_t step = e.control_id - ENCODER_1;
      dev.SendViewSeqStep(step);
      dev.set_play_seq(dev.view_seq());
      dev.set_manual_step_selected(1);
    }
  }

  if (e.value == DBLCLICK) {
    if (e.control_id == ENCODER_A) {
      ;
    } else
    if (e.control_id == ENCODER_B) {
      ;
    }
  }
}

void Ui::HandleSwitchEvent(const Event& e) {
  if (e.control_id == SWITCH) {
    ;
  } else
  if (e.control_id == SWITCH_SHIFTED) {
    ;
  } else
  if (e.control_id == SIDE_SWITCH) {
    switch (e.value) {
      // Left side switches select sequence to play and view
      case SIDE_SWITCH_SEQA: 
      case SIDE_SWITCH_SEQB: 
      case SIDE_SWITCH_SEQC: 
      case SIDE_SWITCH_SEQD: 
        { uint8_t seq = e.value - SIDE_SWITCH_SEQA;
          dev.SetPlaySeq(seq);
          dev.SetViewSeq(seq);
        }
        break;
      // Right side switches perform switch specific actions
      case SIDE_SWITCH_RUN: dev.ToggleRun(); break;
      case SIDE_SWITCH_SAVE: ShowPage(PAGE_SAVE); break;
      case SIDE_SWITCH_LOAD: ShowPage(PAGE_LOAD); break;
      case SIDE_SWITCH_SHIFT: break;
    }
  } else
  if (e.control_id == SIDE_SWITCH_SHIFTED) {
    switch (e.value) {
      // Shifted left side switches select sequence to view
      case SIDE_SWITCH_SEQA: 
      case SIDE_SWITCH_SEQB: 
      case SIDE_SWITCH_SEQC: 
      case SIDE_SWITCH_SEQD: 
        { uint8_t seq = e.value - SIDE_SWITCH_SEQA;
          dev.SetViewSeq(seq);
        }
        break;
      // Shifted right side switches perform switch specific actions
      case SIDE_SWITCH_RUN: dev.ToggleRecording(); break;
      case SIDE_SWITCH_SAVE: break;
      case SIDE_SWITCH_LOAD: break;
      case SIDE_SWITCH_SHIFT: break;
    }
  } else
  if (e.control_id == SIDE_SWITCH_RAISED) {
    ;
  }
}

/* static */
void Ui::HandleRequestEvent(const Event& e) {
  switch (e.control_id) {
    case REQUEST_SHOWPAGE:
      if (active_page_ != e.value) {
        ShowPage(static_cast<UiPageIndex>(e.value));
      } else
        UpdateScreen();
      break;
    case REQUEST_SLOTDATA:
      sysex.SendSlot(e.value);
      break;
#ifdef ENABLE_CV_EXT
    case REQUEST_TUNEDATA:
      sysex.SendTune();
      break;
#endif
  }
}

/* static */
void Ui::HandleUpdateScreen() {
}

/* static */
void Ui::HandleUpdateLeftSideLeds() {
  for (uint8_t n = 0; n < kNumSeqs; n++) {
    uint8_t intensity = 0;

    // The currently viewed sequence LED is fully lit
    if (n == dev.view_seq()) {
      intensity = 0xf;
    } else {
      // The currenly selected for playing sequence group, if any, is dimly lit
      switch (dev.seq_link_mode()) {
        case SEQ_LINK_MODE_16: intensity = (dev.play_seq() & 2) == (n & 2) ? 1 : 0; break;
        case SEQ_LINK_MODE_32: intensity = 1; break;
      }
    }

    // The currently playing sequence is always blinking
    if (dev.running() && n == dev.play_seq()) {
      uint8_t half_step = dev.clock_prescaler() / 2;
      if (intensity == 0xf) {
        intensity = tick_count_ < half_step ? 0xf : 0;
      } else {
        intensity = tick_count_ < half_step ? 0x5 : 0;
      }
    }

    leds.set_pixel(LED_SEQA + n, intensity);
  }
}

/* static */
void Ui::HandleUpdateRightSideLeds() {
  // LED_RUN is updated in step callback
  leds.set_pixel(LED_LOAD, active_page_ == PAGE_LOAD ? 0xf : 0);
  leds.set_pixel(LED_SAVE, active_page_ == PAGE_SAVE ? 0xf : 0);
  leds.set_pixel(LED_SHIFT, IsShifted() ? 0xf : 0);
}

/* static */
void Ui::HandleUpdateStepLeds() {
  for (uint8_t n = 0; n < kNumSteps; n++) {
    uint8_t intensity;
    if (n == dev.step() && (((dev.running() || dev.recording()) && !dev.all_steps_skipped()) || dev.manual_step_selected())) {
      if (dev.manual_step_selected() && !dev.recording()) {
        intensity = 1;
      } else {
        intensity = (dev.view_seq() == dev.play_seq()) ? 0xf : 5;
      }
    } else {
      intensity = dev.recording() ? 1 : 0;
    }

    leds.set_pixel(LED_1 + n, intensity);
  }
}

/* static */
void Ui::ShowPage(UiPageIndex page, uint8_t remember_last_page) {
  // Flush the event queue
  queue_.Flush();
  queue_.Touch();

  // Inform current page if any
  if (*event_handlers_.OnQuit) {
    (*event_handlers_.OnQuit)(page);
  }

  // Only remember primary pages
  if (remember_last_page && active_page_ >= FIRST_PAGE && active_page_ <= LAST_PAGE) {
    last_page_ = active_page_;
  }

  // Activate new page
  UiPageIndex prevPage = active_page_;
  active_page_ = page;

  // Load the page info structure
  ResourcesManager::Load(page_registry, page, &page_info_);

  // Load the event handlers structure
  ResourcesManager::Load(page_info_.event_handlers, 0, &event_handlers_);

  (*event_handlers_.OnInit)(&page_info_, prevPage);

  UpdateScreen();
}

Dialog::Dialog(prog_char* text, prog_char* buttons, uint8_t num_buttons) {
  data_.text = text;
  data_.buttons = buttons;
  data_.num_buttons = num_buttons;
  Ui::ShowDialog(data_);
}

/* static */
void Ui::ShowDialog(DlgData& dlg_data) {
  // Flush the event queue
  queue_.Flush();
  queue_.Touch();

  // Load the Dlg page info structure
  ResourcesManager::Load(page_registry, PAGE_DLG, &page_info_);

  // Load the event handlers structure
  ResourcesManager::Load(page_info_.event_handlers, 0, &event_handlers_);

  memcpy(&page_info_.dlg_data, &dlg_data, sizeof(DlgData));

  (*event_handlers_.OnInit)(&page_info_, active_page_);

  UpdateScreen();

  // Run local message loop
  page_info_.dlg_data.result = kNoDlgResult;
  while (page_info_.dlg_data.result == kNoDlgResult) {
    ui.DoEvents();
  }

  // Set result for the caller
  dlg_data.result = page_info_.dlg_data.result;

  // Flush the event queue.
  queue_.Flush();
  queue_.Touch();

  // Load the active page info structure
  ResourcesManager::Load(page_registry, active_page_, &page_info_);

  // Load the event handlers structure
  ResourcesManager::Load(page_info_.event_handlers, 0, &event_handlers_);

  UpdateScreen();
}

/* static */
void Ui::ShowStepPage(uint8_t step){ 
  StepPage::set_step(step);
  ShowPage(PAGE_STEP);
}

/* static */
void Ui::ClearScreen() {
  display.Clear();
}

/* static */
void Ui::UpdateScreen() {
  ClearScreen();
  if (!(*event_handlers_.UpdateScreen)()) {
    HandleUpdateScreen();
  }
}

/* static */
void Ui::RedrawScreen() {
  for (uint8_t n = 0; n < kLcdWidth * kLcdHeight; n++) {
    display.Tick();
  }
}

/* static */
uint8_t Ui::IsShifted() {
  if (sideSwitches_.low(SIDE_SWITCH_SHIFT)) {
    inhibit_shift_raised_ = 1;
    return 1;
  }
  return 0;
}

/* static */
int8_t Ui::FixOctaveIncrement(int8_t value) {
  if (value > 1) return 12;
  if (value < -1) return -12;
  return value;
}

/* static */
void Ui::PrintHex(char* buffer, uint8_t value) {
  *buffer++ = NibbleToAscii(U8ShiftRight4(value));
  *buffer = NibbleToAscii(value & 0xf);
}

/* static */
void Ui::PrintNote(char* buffer, uint8_t note) {
  uint8_t octave = 0;
  while (note >= 12) {
    ++octave;
    note -= 12;
  }

  static const prog_char note_names[] PROGMEM = "CCDDEFFGGAAB";
  static const prog_char octaves[] PROGMEM = "-0123456789";

  *buffer++ = ResourcesManager::Lookup<char, uint8_t>(
      note_names, note);

  switch (note) {
    case 1: case 3: case 6: case 8: case 10: 
      *buffer++ = '#';
      break;
  }

  *buffer = ResourcesManager::Lookup<char, uint8_t>(
      octaves, octave);
}

/* static */
void Ui::PrintNN(char* buffer, uint16_t numb, char filler) {
  uint8_t x = 0;
  if (numb < 10) { x = 1; buffer[0] = filler; }
  UnsafeItoa(numb, 2 - x, &buffer[x]);
}

/* static */
void Ui::PrintNNN(char* buffer, uint16_t numb, char filler) {
  uint8_t x = 0;
  if (numb < 10)  { x = 2; buffer[0] = filler; buffer[1] = filler; } else 
  if (numb < 100) { x = 1; buffer[0] = filler; }
  UnsafeItoa(numb, 3 - x, &buffer[x]);
}

/* static */
uint8_t Ui::GetEncoderState(uint8_t id) {
  if (id == ENCODER_A) {
    return encoderA_.immediate_value();
  } else
  if (id == ENCODER_B) {
    return encoderB_.immediate_value();
  } else
  if (id >= ENCODER_1 && id <= ENCODER_8) {
    return encoders_.state(id - ENCODER_1);
  }
  return 0;
}

/* static */
uint8_t Ui::GetEncoderClickValue(uint8_t id) {
  uint32_t time = milliseconds(); uint8_t value;
  if (last_click_encoder_ == id) {
    value = (time - last_click_time_) < kDblClickTime ? DBLCLICK : CLICK;
  } else {
    last_click_encoder_ = id;
    value = CLICK;
  }
  
  last_click_time_ = time;
  
  return value  + IsShifted();
}

/* static */
void Ui::RunningCallback() {
  leds.set_pixel(LED_RUN, dev.running() ? 0xf : 0);

  if (prev_running_callback) {
    prev_running_callback();
  }
}

/* static */
void Ui::StepCallback() {
  if (dev.running()) {
    tick_count_ = 0;
    step_blink_count_ = 5;
    leds.set_pixel(LED_RUN, 0xf);
  }
  
  if (prev_step_callback) {
    prev_step_callback();
  }
}

/* static */
void Ui::TickCallback() {
  if (dev.running()) {
    ++tick_count_;
  }
  
  if (prev_tick_callback) {
    prev_tick_callback();
  }
}

/* static */
void Ui::Tick() {
  if (step_blink_count_ > 0 && dev.running()) {
    leds.set_pixel(LED_RUN, --step_blink_count_ ? 0xf : 0);
  }
}

} // namespace midialf
