// Copyright 2012 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// UI page for CCx/send editing

#include "midialf/ui_pages/cc_page.h"

namespace midialf {

/* <static> */
uint8_t CCPage::cc_;
/* </static> */

/* static */
const prog_EventHandlers CCPage::event_handlers_ PROGMEM = {
  OnInit,
  OnQuit,
  OnIncrement,
  OnClick,
  OnSwitch,
  OnIdle,
  UpdateScreen,
  UpdateLeds,
};

/* static */
void CCPage::OnInit(PageInfo* pageInfo, UiPageIndex prevPage) {
  if (prevPage <= LAST_PAGE) {
    cc_ = prevPage < PAGE_CC ? 0 : kNumCCs - 1;
  }
}

/* static */
void CCPage::OnQuit(UiPageIndex nextPage) {
}

/* static */
uint8_t CCPage::OnIncrement(uint8_t id, int8_t value) {
  // ENCA changes all step cc values up/down
  if (id == ENCODER_A) {
    for (uint8_t n = 0; n < kNumSteps; n++) {
      uint8_t cc = view_seq_step_cc(n);
      set_view_seq_step_cc(n, Clamp7F(static_cast<int16_t>(cc) + value));
    }
    return 1;
  }
  // ENCB selects CC page
  if (id == ENCODER_B) {
    if (value > 0) {
      if (cc_ == kNumCCs - 1) {
        return 0;
      } else
        ++cc_;
    } else
    if (value < 0) {
      if (cc_ == 0) {
        return 0;
      } else
        --cc_;
    }
    return 1;
  }
  // ENCx changes step cc value up/down
  if (id >= ENCODER_1 && id <= ENCODER_8) {
    uint8_t step = id - ENCODER_1;
    uint8_t cc = view_seq_step_cc(step);
    set_view_seq_step_cc(step, Clamp7F(static_cast<int16_t>(cc) + value));
    dev.SendViewSeqStep(step);
    return 1;
  }
  return 0;
}

/* static */
uint8_t CCPage::OnClick(uint8_t id, uint8_t value) {
  return 0;
}

/* static */
uint8_t CCPage::OnSwitch(uint8_t id, uint8_t value) {
  if (id == SWITCH && value < kNumSteps) {
    uint8_t cc1send = view_seq_step_send_cc(value);
    set_view_seq_step_send_cc(value, !cc1send);
    return 1;
  }
  return 0;
}

/* static */
uint8_t CCPage::OnIdle() {
  return 0;
}

/* static */
uint8_t CCPage::UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  DrawSeparators();
  DrawStepIcons(line1);

  uint8_t noCC = true;
  for (uint8_t n = 0; n < kNumSteps; n++) {
    uint8_t x = cell_pos(n) + 1;
    UnsafeItoa(view_seq_step_cc(n), 3, &line1[x]);
    if (view_seq_step_send_cc(n)) {
      char* pch = &line2[x];
      *pch++ = 'c';
      *pch++ = 'c';
      *pch++ = '1' + cc_;
      noCC = false;
    }
  }
  if (noCC) {
    static const prog_char noCC[] PROGMEM = "No CC";
    uint8_t x = kLcdWidth / 2 - lengof(noCC) / 2 - 1;
    memcpy_P(&line2[x], noCC, lengof(noCC));
    line2[x + lengof(noCC)] = '1' + cc_;
  }

  return 1;
}

/* static */
uint8_t CCPage::UpdateLeds() {
  return 0;
}

/* static */
uint8_t CCPage::view_seq_step_cc(uint8_t step) {
  switch (cc_) {
    case 0: return dev.view_seq_step_cc1(step);
    case 1: return dev.view_seq_step_cc2(step);
  }
  return 0;
}

/* static */
void CCPage::set_view_seq_step_cc(uint8_t step, uint8_t cc) {
  switch (cc_) {
    case 0: dev.set_view_seq_step_cc1(step, cc); break;
    case 1: dev.set_view_seq_step_cc2(step, cc); break;
  }
}

/* static */
uint8_t CCPage::view_seq_step_send_cc(uint8_t step) {
  switch (cc_) {
    case 0: return dev.view_seq_step_send_cc1(step);
    case 1: return dev.view_seq_step_send_cc2(step);
  }
  return 0;
}

/* static */
void CCPage::set_view_seq_step_send_cc(uint8_t step, uint8_t send_cc) {
  switch (cc_) {
    case 0: dev.set_view_seq_step_send_cc1(step, send_cc); break;
    case 1: dev.set_view_seq_step_send_cc2(step, send_cc); break;
  }
}

} // namespace midialf
