// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Command page.

#include "midialf/ui_pages/command_page.h"
#include "midialf/storage.h"
#include "midialf/scale.h"
#include "midialf/sysex.h"

#include "avrlib/random.h"

namespace midialf {

/* <static> */
uint8_t CommandPage::command_;
uint32_t CommandPage::started_time_;
uint8_t CommandPage::scale_;
uint8_t CommandPage::slot_;
/* </static> */

/* static */
const prog_EventHandlers CommandPage::event_handlers_ PROGMEM = {
  OnInit,
  OnQuit,
  OnIncrement,
  OnClick,
  OnSwitch,
  OnIdle,
  UpdateScreen,
  UpdateLeds,
};

/* static */
void CommandPage::OnInit(PageInfo* pageInfo, UiPageIndex prevPage) {
  started_time_ = milliseconds();
  Random::Seed(started_time_);
  slot_ = dev.slot();
}

/* static */
void CommandPage::OnQuit(UiPageIndex nextPage) {
}

/* static */
uint8_t CommandPage::OnIncrement(uint8_t id, int8_t value) {
  // ENCA selects a command or shifts notes up/down
  if (id == ENCODER_A) {
    if (ui.IsShifted()) {
      value = Ui::FixOctaveIncrement(value);
      for (uint8_t n = 0; n < kNumSteps; n++) {
        uint8_t note = dev.view_seq_step_note(n);
        dev.set_view_seq_step_note(n, Clamp7F(static_cast<int16_t>(note) + value));
      }
    } else {
      if (value > 0) {
        if (command_ < kNumCommands - 1) {
          ++command_;
          started_time_ = milliseconds();
        }
      } else
      if (value < 0) {
        if (command_ > 0) {
          --command_;
          started_time_ = milliseconds();
        }
      }
    }
    return 1;
  }

  // ENCB selects scale for randomize command
  if (id == ENCODER_B) {
    switch (command_) {
      case copySeq:
        break;
      case rotateSeq:
        break;
      case randomizeSeq:
        scale_ = Clamp(static_cast<int16_t>(scale_) + value, 0, kNumScales - 1);
        break;
      case initializeSeq:
      case sysexOneSlot:
        slot_ = Clamp(static_cast<int16_t>(slot_) + value, 0, storage.num_slots() - 1);
        break;
      case sysexAllSlots:
        break;
#ifdef ENABLE_CV_EXT
      case sysexTuneData:
      case tuneNoteCv:
        break;
#endif
    }
    return 1;
  }

  // ENCx scales the step for randomize command
  if (id >= ENCODER_1 && id <= ENCODER_8) {
    switch (command_) {
      case copySeq:
        break;
      case rotateSeq:
        break;
      case randomizeSeq:
        { uint8_t step = id - ENCODER_1;
          uint8_t note = dev.view_seq_step_note(step);
          if (value == -1 || value == 1) {
            note = Scale::GetNextScaledNote(scale_, note, value > 0);
          } else {
            value = Ui::FixOctaveIncrement(value);
            note = Clamp7F(static_cast<int16_t>(note) + value);
          }
          dev.set_view_seq_step_note(step, note);
          dev.SendViewSeqStep(step);
        }
        break;
      case initializeSeq:
      case sysexOneSlot:
      case sysexAllSlots:
        break;
#ifdef ENABLE_CV_EXT
      case sysexTuneData:
      case tuneNoteCv:
        break;
#endif
    }
    return 1;
  }

  return 0;
}

/* static */
uint8_t CommandPage::OnClick(uint8_t id, uint8_t value) {
  if (value != CLICK)
    return 0;

  // ENCA terminates command mode
  if (id == ENCODER_A) {
    ui.ShowLastPage();
    return 1;
  }

  // ENCB randomizes and initializes selected seqence
  if (id == ENCODER_B) {
    switch (command_) {
      case copySeq:
        break;
      case rotateSeq:
        break;
      case randomizeSeq:
        dev.RandomizeSeq(dev.view_seq(), scale_);
        break;
      case initializeSeq:
        dev.InitSeq(dev.view_seq());
        break;
      case sysexOneSlot:
        ui.RequestShowPage(PAGE_SYSEX);
        sysex.SendSlot(slot_);
        break;
      case sysexAllSlots:
        ui.RequestShowPage(PAGE_SYSEX);
        sysex.SendAllSlots();
        break;
#ifdef ENABLE_CV_EXT
      case sysexTuneData:
        ui.RequestShowPage(PAGE_SYSEX);
        sysex.SendTune();
        break;
      case tuneNoteCv:
        ui.RequestShowPage(PAGE_TUNE);
        break;
#endif
    }
    return 1;
  }

  return 0;
}

/* static */
uint8_t CommandPage::OnSwitch(uint8_t id, uint8_t value) {
  switch (command_) {
    case copySeq:
      if (id == SIDE_SWITCH && value >= SIDE_SWITCH_SEQA && value <= SIDE_SWITCH_SEQD) {
        uint8_t seq = value - SIDE_SWITCH_SEQA;
        dev.CopySeq(dev.view_seq(), seq);
        dev.SetSeq(seq);
        return 1;
      }
      break;
    case rotateSeq:
      if (id == SWITCH && value >= SWITCH_1 && value <= SWITCH_8) {
        uint8_t step = value - SWITCH_1;
        dev.RotateSeq(dev.view_seq(), step);
        return 1;
      }
      break;
    case randomizeSeq:
      if (id == SWITCH && value >= SWITCH_1 && value <= SWITCH_8) {
        uint8_t step = value - SWITCH_1;
        dev.RandomizeSeqStep(dev.view_seq(), step, scale_);
        dev.SendViewSeqStep(step);
        return 1;
      }
      break;
    case initializeSeq:
      if (id == SWITCH && value >= SWITCH_1 && value <= SWITCH_8) {
        uint8_t step = value - SWITCH_1;
        dev.InitSeqStep(dev.view_seq(), step);
        return 1;
      }
      break;
    case sysexOneSlot:
    case sysexAllSlots:
      break;
#ifdef ENABLE_CV_EXT
    case sysexTuneData:
    case tuneNoteCv:
      break;
#endif
  }

  return 0;
}

/* static */
uint8_t CommandPage::OnIdle() {
  return 0;
}

/* static */
uint8_t CommandPage::UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  switch (command_) {
    case copySeq:
      { static const prog_char prompt[] PROGMEM = "Copy SeqX:";
        memcpy_P(&line1[0], prompt, lengof(prompt));
        line1[lengof(prompt) - 2] = 'A' + dev.view_seq();

        DrawNotes(line2);
      }
      break;
    case rotateSeq:
      { static const prog_char prompt[] PROGMEM = "Rotate SeqX:";
        memcpy_P(&line1[0], prompt, lengof(prompt));
        line1[lengof(prompt) - 2] = 'A' + dev.view_seq();
        DrawNotes(line2);
      }
      break;
    case randomizeSeq:
      { static const prog_char prompt[] PROGMEM = "Randomize SeqX:";
        memcpy_P(&line1[0], prompt, lengof(prompt));
        line1[lengof(prompt) - 2] = 'A' + dev.view_seq();
        uint8_t x = kLcdWidth / 2 - 4;
        Ui::PrintNNN(&line1[x], 1 + scale_);
        Scale::GetScaleName(scale_, (uint8_t*)&line1[x + 4]);
        DrawNotes(line2);
      }
      break;
    case initializeSeq:
      { static const prog_char prompt[] PROGMEM = "Initialize SeqX:";
        memcpy_P(&line1[0], prompt, lengof(prompt));
        line1[lengof(prompt) - 2] = 'A' + dev.view_seq();
        DrawNotes(line2);
      }
      break;
    case sysexOneSlot:
      { static const prog_char prompt[] PROGMEM = "Sysex slot: ";
        memcpy_P(&line1[0], prompt, lengof(prompt));
        Ui::PrintNNN(&line1[lengof(prompt)], 1 + slot_);
        Slot::ReadName(slot_, (uint8_t*)&line1[lengof(prompt) + 4]);
        goto Send;
      }
      break;
    case sysexAllSlots:
      { static const prog_char prompt[] PROGMEM = "Sysex all slots";
        memcpy_P(&line1[0], prompt, lengof(prompt));
Send:   static const prog_char cmdSend[] PROGMEM = "[Send]";
        memcpy_P(&line1[kLcdWidth - lengof(cmdSend)], cmdSend, lengof(cmdSend));
      }
      break;
#ifdef ENABLE_CV_EXT
    case sysexTuneData:
      { static const prog_char prompt[] PROGMEM = "Sysex note CVs tune data";
        memcpy_P(&line1[0], prompt, lengof(prompt));
        goto Send;
      }
      break;
    case tuneNoteCv:
      { static const prog_char prompt[] PROGMEM = "Tune note CVs";
        memcpy_P(&line1[0], prompt, lengof(prompt));
        static const prog_char cmdTune[] PROGMEM = "[Tune]";
        memcpy_P(&line1[kLcdWidth - lengof(cmdTune)], cmdTune, lengof(cmdTune));
      }
      break;
#endif
  }

  return 1;
}

/* static */
uint8_t CommandPage::UpdateLeds() {
  switch (command_) {

    case copySeq:
      { uint8_t intensity = (milliseconds() - started_time_) % 200 > 100 ? 0 : 0xf;
        for (uint8_t n = 0; n < kNumSeqs; n++) {
          if (n == dev.view_seq()) {
            leds.set_pixel(LED_SEQA + n, 0xf);
          } else {
            leds.set_pixel(LED_SEQA + n, intensity);
          }
        }
      }
      ui.HandleUpdateRightSideLeds();
      ui.HandleUpdateStepLeds();
      return 1;

    case rotateSeq:
    case randomizeSeq:
    case initializeSeq:
      { uint8_t intensity = (milliseconds() - started_time_) % 200 > 100 ? 0 : 0x1;
        for (uint8_t n = 0; n < kNumSteps; n++) {
          if (dev.running() && n == dev.step() && (dev.view_seq() == dev.play_seq())) {
            leds.set_pixel(LED_1 + n, 0xf);
          } else {
            leds.set_pixel(LED_1 + n, intensity);
          }
        }
      }
      ui.HandleUpdateLeftSideLeds();
      ui.HandleUpdateRightSideLeds();
      return 1;
    case sysexOneSlot:
    case sysexAllSlots:
      break;
  }
  return 0;
}

/* static */
void CommandPage::DrawNotes(char* line) {
  DrawSeparators(line);
  for (uint8_t n = 0; n < kNumSteps; n++) {
    Ui::PrintNote(&line[cell_pos(n) + 1], dev.view_seq_step_note(n));
  }
}

} // namespace midialf
