// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Configuration pages.

#include "midialf/ui_pages/config_page.h"
#include "midialf/clock.h"

#include <stdlib.h>

namespace midialf {

/* <static> */
uint8_t ConfigPage::page_;
uint8_t ConfigPage::num_taps_;
uint32_t ConfigPage::elapsed_time_;
uint8_t ConfigPage::learning_;
ConfigPage::Prev* ConfigPage::prev_;
/* </static> */

/* static */
const prog_EventHandlers ConfigPage::event_handlers_ PROGMEM = {
  OnInit,
  OnQuit,
  OnIncrement,
  OnClick,
  OnSwitch,
  OnIdle,
  UpdateScreen,
  UpdateLeds,
};

/* static */
void ConfigPage::OnInit(PageInfo* pageInfo, UiPageIndex prevPage) {
  if (prevPage <= LAST_PAGE) {
    page_ = prevPage < PAGE_CONFIG ? 0 : kNumPages - 1;
  }

  prev_ = (Prev*)malloc(sizeof(Prev));

  InitPage();

  dev.set_note_callback(NoteCallback);
  dev.set_control_change_callback(ControlChangeCallback);
}

/* static */
void ConfigPage::OnQuit(UiPageIndex nextPage) {
  free(prev_);

  dev.set_note_callback(0);
  dev.set_control_change_callback(0);
}

/* static */
uint8_t ConfigPage::OnIncrement(uint8_t id, int8_t value) {
  // ENCB selects config page
  if (id == ENCODER_B) {
    if (value > 0) {
      if (page_ == kNumPages - 1) {
        return 0;
      } else {
        ++page_;
        InitPage();
      }
    } else
    if (value < 0) {
      if (page_ == 0) {
        return 0;
      } else {
        --page_;
        InitPage();
      }
    }
    return 1;
  }

  switch (page_) {
    case 0: return HandlePage1_OnIncrement(id, value);
    case 1: return HandlePage2_OnIncrement(id, value);
    case 2: return HandlePage3_OnIncrement(id, value);
    case 3: return HandlePage4_OnIncrement(id, value);
#ifdef ENABLE_CV_EXT
    case 4: return HandlePage5_OnIncrement(id, value);
#endif
  }
  return 1;
}

#define DEV_SET(name, Name) \
  dev.set_##name(Clamp(static_cast<int16_t>(dev.name()) + value, kMin##Name, kMax##Name))

#define DEV_SET_16(name, Name) \
  dev.set_##name(Clamp16(static_cast<int16_t>(dev.name()) + value, kMin##Name, kMax##Name))

#define DEV_SET_INDEX(name, Name, index) \
  dev.set_##name(index, Clamp(static_cast<int16_t>(dev.name(index)) + value, kMin##Name, kMax##Name))

#define DEV_HAS_FLAG(Flag) \
  dev.flags() & Flag

#define DEV_SET_FLAG(Flag, value) \
  dev.set_flags(value ? dev.flags() | Flag : dev.flags() & ~Flag)

#define DEV_SET_WITH_FLAG(name, Name, Flag) \
  if (DEV_HAS_FLAG(Flag)) { \
    if (value < 0 && dev.name() == kMin##Name) { \
      DEV_SET_FLAG(Flag, 0); \
    } else \
      DEV_SET(name, Name); \
  } else { \
    if (!(value < 0 && dev.name() == kMin##Name)) { \
      DEV_SET_FLAG(Flag, 1); \
    } \
  }

/* static */
uint8_t ConfigPage::HandlePage1_OnIncrement(uint8_t id, int8_t value) {
  switch (id) {
    case ENCODER_1: dev.SetChannel(Clamp(static_cast<int16_t>(dev.channel()) + value, 0, 0xf)); break;
    case ENCODER_2: DEV_SET(tempo, Tempo); break;
    case ENCODER_3: DEV_SET(step_length, StepLength); break;
    case ENCODER_4: DEV_SET_16(swing, Swing); break;
    case ENCODER_5: value = Ui::FixOctaveIncrement(value); DEV_SET(seq_root_note, SeqRootNote); break;
    case ENCODER_6: DEV_SET(seq_transpose_mode, SeqTransposeMode); break;
    case ENCODER_7: DEV_SET(seq_switch_mode, SeqSwitchMode); break;
    case ENCODER_8: DEV_SET(seq_link_mode, SeqLinkMode); break;
  }
  return 1;
}

/* static */
uint8_t ConfigPage::HandlePage2_OnIncrement(uint8_t id, int8_t value) {
  switch (id) {
    case ENCODER_1: DEV_SET(seq_direction, SeqDirection); break;
    case ENCODER_2: DEV_SET(steps_forward, StepCount); break;
    case ENCODER_3: DEV_SET(steps_backward, StepCount); break;
    case ENCODER_4: DEV_SET(steps_replay, StepCount); break;
    case ENCODER_5: DEV_SET(steps_interval, StepCount); break;
    case ENCODER_6: DEV_SET(steps_repeat, StepCount); break;
    case ENCODER_7: DEV_SET(steps_skip, StepCount); break;
    case ENCODER_8: DEV_SET_FLAG(kSongModeActive, value > 0); break;
  }
  return 1;
}

/* static */
uint8_t ConfigPage::HandlePage3_OnIncrement(uint8_t id, int8_t value) {
  switch (id) {
    case ENCODER_1: DEV_SET(cc1_number, CCNumber); break;
    case ENCODER_2: DEV_SET(cc2_number, CCNumber); break;
    case ENCODER_3: break;
    case ENCODER_4: break;
    case ENCODER_5: SetClockModeFlag(CLOCK_MODE_EXTERNAL, value > 0); break;
    case ENCODER_6: SetNextClockOutput(/* forward */ value > 0, /* cycle */ 0); break;
    case ENCODER_7: break;
#ifdef ENABLE_CV_EXT
    case ENCODER_8: DEV_SET(strobe_width, StrobeWidth); break;
#else
    case ENCODER_8: break;
#endif
  }
  return 1;
}

/* static */
uint8_t ConfigPage::HandlePage4_OnIncrement(uint8_t id, int8_t value) {
  switch (id) {
    case ENCODER_1: DEV_SET_WITH_FLAG(bank_select_msb, BankSelectMsb, kSendBankSelectMsb); dev.SendSeqProgramChange(); break;
    case ENCODER_2: DEV_SET_WITH_FLAG(bank_select_lsb, BankSelectLsb, kSendBankSelectLsb); dev.SendSeqProgramChange(); break;
    case ENCODER_3: DEV_SET_WITH_FLAG(program_change, ProgramChange, kSendProgChange); dev.SendSeqProgramChange(); break;
    case ENCODER_4: break;
    case ENCODER_5: DEV_SET(prog_change_mode, ProgChangeMode); break;
    case ENCODER_6: DEV_SET(ctrl_change_mode, CtrlChangeMode); break;
    case ENCODER_7: break;
    case ENCODER_8: break;
  }
  return 1;
}

#ifdef ENABLE_CV_EXT

/* static */
uint8_t ConfigPage::HandlePage5_OnIncrement(uint8_t id, int8_t value) {
  switch (id) {
    case ENCODER_1:
    case ENCODER_2:
    case ENCODER_3:
    case ENCODER_4: DEV_SET_INDEX(cv_mode, CVMode, id - ENCODER_1); break;
    case ENCODER_5: 
    case ENCODER_6: 
    case ENCODER_7: 
    case ENCODER_8: DEV_SET_INDEX(gate_mode, GateMode, id - ENCODER_5); break; 
      break;
  }
  return 1;
}

#endif // #ifdef ENABLE_CV_EXT

#undef DEV_SET
#undef DEV_SET_16
#undef DEV_SET_INDEX
#undef DEV_HAS_FLAG
#undef DEV_SET_FLAG
#undef DEV_SET_WITH_FLAG

/* static */
uint8_t ConfigPage::OnClick(uint8_t id, uint8_t value) {
  // ENCA shows command page
  if (id == ENCODER_A) {
    return 0;
  }
  // ENCB shows previous page
  if (id == ENCODER_B) {
    if (ui.last_page() < PAGE_CONFIG) {
      ui.ShowLastPage();
    } else {
      ui.ShowPage(FIRST_PAGE);
    }
    return 1;
  }
  return 1;
}

/* static */
uint8_t ConfigPage::OnSwitch(uint8_t id, uint8_t value) {
  if (id != SWITCH)
    return 0;

  switch (page_) {
    case 0: return HandlePage1_OnSwitch(id, value);
    case 1: return HandlePage2_OnSwitch(id, value);
    case 2: return HandlePage3_OnSwitch(id, value);
    case 3: return HandlePage4_OnSwitch(id, value);
#ifdef ENABLE_CV_EXT
    case 4: return HandlePage5_OnSwitch(id, value);
#endif
  }
  return 1;
}

#define CYCLE_SETTING(name, Name) \
  dev.set_##name(dev.name() < kMax##Name ? dev.name() + 1 : kMin##Name)

#define TOGGLE_SETTING_INDEX(name, index) \
  dev.set_##name(index, dev.name(index) ? 0 : 1)

#define TOGGLE_DEF_PREV(name, Name) \
  dev.set_##name(dev.name() != kDef##Name ? kDef##Name : prev_->name)

#define TOGGLE_DEV_FLAG(name, Flag) \
  dev.set_##name(dev.name() & Flag ? dev.name() & ~Flag : dev.name() | Flag)

#define TOGGLE_LEARNING(name) \
  learning_ = (learning_ == learning##name) ? noLearning : learning##name

/* static */
uint8_t ConfigPage::HandlePage1_OnSwitch(uint8_t id, uint8_t value) {
  switch (value) {
    case SWITCH_1: TOGGLE_LEARNING(Channel); break;
    case SWITCH_2: UpdateTapTempo(); break;
    case SWITCH_3: TOGGLE_DEF_PREV(step_length, StepLength); break;
    case SWITCH_4: TOGGLE_DEF_PREV(swing, Swing); break;
    case SWITCH_5: TOGGLE_LEARNING(SeqRootNote); break;
    case SWITCH_6: CYCLE_SETTING(seq_transpose_mode, SeqTransposeMode); break;
    case SWITCH_7: CYCLE_SETTING(seq_switch_mode, SeqSwitchMode); break;
    case SWITCH_8: CYCLE_SETTING(seq_link_mode, SeqLinkMode); break;
  }
  return 1;
}

/* static */
uint8_t ConfigPage::HandlePage2_OnSwitch(uint8_t id, uint8_t value) {
  switch (value) {
    case SWITCH_1: CYCLE_SETTING(seq_direction, SeqDirection); break;
    case SWITCH_2: TOGGLE_DEF_PREV(steps_forward, StepCount); break;
    case SWITCH_3: TOGGLE_DEF_PREV(steps_backward, StepCount); break;
    case SWITCH_4: TOGGLE_DEF_PREV(steps_replay, StepCount); break;
    case SWITCH_5: TOGGLE_DEF_PREV(steps_interval, StepCount); break;
    case SWITCH_6: TOGGLE_DEF_PREV(steps_repeat, StepCount); break;
    case SWITCH_7: TOGGLE_DEF_PREV(steps_skip, StepCount); break;
    case SWITCH_8: TOGGLE_DEV_FLAG(flags, kSongModeActive); break;
  }
  return 1;
}

/* static */
uint8_t ConfigPage::HandlePage3_OnSwitch(uint8_t id, uint8_t value) {
  switch (value) {
    case SWITCH_1: TOGGLE_LEARNING(CC1Number); break;
    case SWITCH_2: TOGGLE_LEARNING(CC2Number); break;
    case SWITCH_3: break;
    case SWITCH_4: break;
    case SWITCH_5: SetClockModeFlag(CLOCK_MODE_EXTERNAL, !dev.ExternalClock()); break;
    case SWITCH_6: SetNextClockOutput(/* forward */ 1, /* cycle */ 1); break;
    case SWITCH_7: break;
#ifdef ENABLE_CV_EXT
    case SWITCH_8: TOGGLE_DEF_PREV(strobe_width, StrobeWidth); break;
#else
    case SWITCH_8: break;
#endif
  }
  return 1;
}

/* static */
uint8_t ConfigPage::HandlePage4_OnSwitch(uint8_t id, uint8_t value) {
  switch (value) {
    case SWITCH_1: TOGGLE_DEV_FLAG(flags, kSendBankSelectMsb); break;
    case SWITCH_2: TOGGLE_DEV_FLAG(flags, kSendBankSelectLsb); break;
    case SWITCH_3: TOGGLE_DEV_FLAG(flags, kSendProgChange); break;
    case SWITCH_4: break;
    case SWITCH_5: TOGGLE_DEF_PREV(prog_change_mode, ProgChangeMode); break;
    case SWITCH_6: TOGGLE_DEF_PREV(ctrl_change_mode, CtrlChangeMode); break;
    case SWITCH_7: break;
    case SWITCH_8: break;
  }
  return 1;
}

#ifdef ENABLE_CV_EXT

/* static */
uint8_t ConfigPage::HandlePage5_OnSwitch(uint8_t id, uint8_t value) {
  switch (value) {
    case SWITCH_1: 
    case SWITCH_2: 
    case SWITCH_3: 
    case SWITCH_4: TOGGLE_SETTING_INDEX(cv_mode_offset, value - SWITCH_1); break;
    case SWITCH_5: 
    case SWITCH_6: 
    case SWITCH_7: 
    case SWITCH_8: TOGGLE_SETTING_INDEX(gate_mode_invert, value - SWITCH_5); break;;
  }
  return 1;
}

#endif // #ifdef ENABLE_CV_EXT

#undef CYCLE_SETTING
#undef TOGGLE_SETTING
#undef TOGGLE_DEF_PREV
#undef TOGGLE_DEV_FLAG
#undef TOGGLE_LEARNING

/* static */
uint8_t ConfigPage::OnIdle() {
  return 0;
}

/* static */
uint8_t ConfigPage::UpdateScreen() {
  DrawSeparators();

  switch (page_) {
    case 0: HandlePage1_UpdateScreen(); break;
    case 1: HandlePage2_UpdateScreen(); break;
    case 2: HandlePage3_UpdateScreen(); break;
    case 3: HandlePage4_UpdateScreen(); break;
#ifdef ENABLE_CV_EXT
    case 4: HandlePage5_UpdateScreen(); break;
#endif
  }
  return 1;
}

/* static */
void ConfigPage::HandlePage1_UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  DrawCells(line1, PSTR("ChanTempStepSwngRootTranXseqLink"));  

  // Channel
  UnsafeItoa(dev.channel() + 1, 3, &line2[cell_pos(0) + 1]);

  if (learning_ == learningChannel) {
    DrawBrackets(&line2[cell_pos(0)], 2);
  }

  // Tempo
  UnsafeItoa(dev.tempo(), 3, &line2[cell_pos(1) + 1]);

  if (num_taps_ > 0) {
    DrawBrackets(&line2[cell_pos(1)], 3);
  }
  
  // Step length

  DrawSelStrN(&line2[cell_pos(2) + 1], dev.step_length(), midi_clock_ticks_per_note_str, kNoteDurationStrLen);

  // Swing
  { uint8_t x = cell_pos(3);
    int16_t swing_int = dev.swing() / 10;
    x += UnsafeItoaLen(swing_int, 3, &line2[x]); line2[x++] = '.';
    x += UnsafeItoaLen(dev.swing() - swing_int * 10, 1, &line2[x]);
  }

  // Sequence root note
  ui.PrintNote(&line2[cell_pos(4) + 1], dev.seq_root_note());

  if (learning_ == learningSeqRootNote) {
    DrawBrackets(&line2[cell_pos(4)], 3);
  }

  // Sequence transpose mode
  DrawSelStr4(&line2[cell_pos(5)], dev.seq_transpose_mode(), PSTR("nonenote"));

  // Sequence switch mode
  DrawSelStr4(&line2[cell_pos(6)], dev.seq_switch_mode(), PSTR(" imm end"));

  // Sequence link mode
  DrawSelStr4(&line2[cell_pos(7)], dev.seq_link_mode(), PSTR(" no 2x161x32"));
}

/* static */
void ConfigPage::HandlePage2_UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  DrawCells(line1, PSTR(" DirForwBackReplItrvReptSkipSong"));

  DrawSelStr4(&line2[cell_pos(0)], dev.seq_direction(), PSTR("forwbackpendrand"));
  UnsafeItoa(dev.steps_forward(), 3, &line2[cell_pos(1) + 1]);
  UnsafeItoa(dev.steps_backward(), 3, &line2[cell_pos(2) + 1]);
  UnsafeItoa(dev.steps_replay(), 3, &line2[cell_pos(3) + 1]);
  UnsafeItoa(dev.steps_interval(), 3, &line2[cell_pos(4) + 1]);
  UnsafeItoa(dev.steps_repeat(), 3, &line2[cell_pos(5) + 1]);
  UnsafeItoa(dev.steps_skip(), 3, &line2[cell_pos(6) + 1]);
  DrawSelStr4(&line2[cell_pos(7)], dev.flags() ? 1 : 0, pchOffOn);
}

/* static */
void ConfigPage::HandlePage3_UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

#ifdef ENABLE_CV_EXT
  DrawCells(line1, PSTR(" CC1 CC2-------- ClkCOut----Strb"));
#else
  DrawCells(line1, PSTR(" CC1 CC2-------- ClkCOut--------"));
#endif

  // CC numbers
  UnsafeItoa(dev.cc1_number(), 3, &line2[cell_pos(0) + 1]);
  UnsafeItoa(dev.cc2_number(), 3, &line2[cell_pos(1) + 1]);

  switch (learning_) {
    case learningCC1Number: DrawBrackets(&line2[cell_pos(0)], 3); break;
    case learningCC2Number: DrawBrackets(&line2[cell_pos(1)], 3); break;
  }

  // Clock source
  DrawSelStr4(&line2[cell_pos(4)], dev.InternalClock() ? 0 : 1, PSTR(" int ext"));

  // Clock output
  DrawSelStr4(&line2[cell_pos(5)], dev.clock_mode() >> 1, PSTR("nonewrun????alws"));

#ifdef ENABLE_CV_EXT
  // Strob width
  DrawSelStr4(&line2[cell_pos(7)], dev.strobe_width(), PSTR(" 1ms 2ms 3ms 4ms 5ms"));
#endif
}

/* static */
void ConfigPage::HandlePage4_UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  DrawCells(line1, PSTR("BnkMBnkLPgmC----PrgCCtlC--------"));  

  uint8_t x;

  // Bank Select MSB
  x = cell_pos(0) + 1;
  if (dev.flags() & kSendBankSelectMsb) {
    UnsafeItoa(dev.bank_select_msb(), 3, &line2[x]);
  } else {
    Draw2Dashes(&line2[x]);
  }

  // Bank Select LSB
  x = cell_pos(1) + 1;
  if (dev.flags() & kSendBankSelectLsb) {
    UnsafeItoa(dev.bank_select_lsb(), 3, &line2[x]);
  } else {
    Draw2Dashes(&line2[x]);
  }

  // Program Change
  x = cell_pos(2) + 1;
  if (dev.flags() & kSendProgChange) {
    UnsafeItoa(dev.program_change(), 3, &line2[x]);
  } else {
    Draw2Dashes(&line2[x]);
  }

  // Send Program and Control Change
  static const prog_char text[] PROGMEM = "nonerecvsendboth";
  DrawSelStr4(&line2[cell_pos(4)], dev.prog_change_mode(), text);
  DrawSelStr4(&line2[cell_pos(5)], dev.ctrl_change_mode(), text);
}

#ifdef ENABLE_CV_EXT

/* static */
void ConfigPage::HandlePage5_UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  DrawCells(line1, PSTR(" CV1 CV2 CV3 CV4 Gt1 Gt2 Gt3 Gt4"));

  for (uint8_t n = 0; n < 4; n++) {
    line1[cell_pos(n)] = dev.cv_mode_offset(n) ? '-' : ' ';
    line1[cell_pos(n + 4)] = dev.gate_mode_invert(n) ? '-' : ' ';
    DrawSelStr4(&line2[cell_pos(n)], dev.cv_mode(n), PSTR("notevelo cc1 cc2lfo1lfo2atchpbnd"));
    DrawSelStr4(&line2[cell_pos(n + 4)], dev.gate_mode(n), PSTR("gatestrbclckstrt seqlfo1lfo2"));
  }
}

#endif // #ifdef ENABLE_CV_EXT

/* static */
uint8_t ConfigPage::UpdateLeds() {
  // Reset tap tempo after some time
  if (num_taps_ > 0 && clock.value() > 400000L) {
    num_taps_ = 0;
    ui.request_redraw();
  }
  
  return 0;
}

/* static */
void ConfigPage::InitPage() {
  num_taps_ = 0;
  learning_ = noLearning;

  prev_->step_length = dev.step_length();
  prev_->swing = dev.swing();
  prev_->seq_direction = dev.seq_direction();
  prev_->steps_forward = dev.steps_forward();
  prev_->steps_backward = dev.steps_backward();
  prev_->steps_replay = dev.steps_replay();
  prev_->steps_interval = dev.steps_interval();
  prev_->steps_repeat = dev.steps_repeat();
  prev_->steps_skip = dev.steps_skip();
  prev_->prog_change_mode = dev.prog_change_mode();
  prev_->ctrl_change_mode = dev.ctrl_change_mode();
#ifdef ENABLE_CV_EXT
  prev_->strobe_width = dev.strobe_width();
#endif
}

/* static */
void ConfigPage::UpdateTapTempo() {
  // Tap bpm code shamlessly stolen from Mutable Instruments MidiPal ClockSource app
  // by Olivier Gillet (ol.gillet@gmail.com)
  uint32_t t = clock.value();
  clock.Reset();
  if (num_taps_ > 0 && t < 400000L) {
    elapsed_time_ += t;
    dev.set_tempo(avrlib::Clip(18750000 * num_taps_ / elapsed_time_, kMinTempo, kMaxTempo));
  } else {
    num_taps_ = 0;
    elapsed_time_ = 0;
  }
  ++num_taps_;
}

/* static */
void ConfigPage::SetClockModeFlag(uint8_t flag, uint8_t value) {
  uint8_t clock_mode = dev.clock_mode();
  SETFLAGTO(clock_mode, flag, value);
  dev.set_clock_mode(clock_mode);
}

/* static */
void ConfigPage::SetNextClockOutput(uint8_t forward, uint8_t cycle) {
  uint8_t clock_mode = dev.clock_mode();

  if (forward) {
    switch (clock_mode & ~CLOCK_MODE_EXTERNAL) {
      case 0: // never -> while running
        clock_mode|= CLOCK_MODE_SENDOUTPUT; 
        break;
      case CLOCK_MODE_SENDOUTPUT: // while running -> always
        clock_mode|= CLOCK_MODE_SENDOUTPUT | CLOCK_MODE_CONTIGUOUS;
        break;
      case CLOCK_MODE_SENDOUTPUT | CLOCK_MODE_CONTIGUOUS: // always -> never
        if (!cycle) return;
        clock_mode&= ~(CLOCK_MODE_SENDOUTPUT | CLOCK_MODE_CONTIGUOUS);
        break;
    }
  } else {
    switch (clock_mode & ~CLOCK_MODE_EXTERNAL) {
      case 0: // never -> always
        if (!cycle) return;
        clock_mode|= CLOCK_MODE_SENDOUTPUT | CLOCK_MODE_CONTIGUOUS;
        break;
      case CLOCK_MODE_SENDOUTPUT | CLOCK_MODE_CONTIGUOUS: // always -> while running
        clock_mode&= ~CLOCK_MODE_CONTIGUOUS; 
        break;
      case CLOCK_MODE_SENDOUTPUT: // while running -> never
        clock_mode&= ~CLOCK_MODE_SENDOUTPUT;
        break;
    }
  }

  dev.set_clock_mode(clock_mode); 
}

/* static */
uint8_t ConfigPage::NoteCallback(uint8_t channel, uint8_t note, uint8_t velocity) {
  switch (learning_) {
    case learningChannel:
      { dev.SetChannel(channel);
        ui.request_redraw();
        learning_ = noLearning;
      }
      break;
    case learningSeqRootNote:
      if (channel == dev.channel()) {
        dev.set_seq_root_note(note);
        ui.request_redraw();
        learning_ = noLearning;
      }
      break;
  }

  return 0;
}

/* static */
uint8_t ConfigPage::ControlChangeCallback(uint8_t channel, uint8_t controller, uint8_t value) {
  switch (learning_) {
    case learningCC1Number:
      if (channel == dev.channel()) {
        dev.set_cc1_number(controller);
        ui.request_redraw();
        learning_ = noLearning;
      }
      break;
    case learningCC2Number:
      if (channel == dev.channel()) {
        dev.set_cc2_number(controller);
        ui.request_redraw();
        learning_ = noLearning;
      }
      break;
  }

  return 0;
}

} // namespace midialf
