// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// UI page for gate/legato editing

#include "midialf/ui_pages/gate_page.h"
#include "midialf/duration.h"

namespace midialf {

/* static */
const prog_EventHandlers GatePage::event_handlers_ PROGMEM = {
  OnInit,
  OnQuit,
  OnIncrement,
  OnClick,
  OnSwitch,
  OnIdle,
  UpdateScreen,
  UpdateLeds,
};

/* static */
void GatePage::OnInit(PageInfo* pageInfo, UiPageIndex prevPage) {
}

/* static */
void GatePage::OnQuit(UiPageIndex nextPage) {
}

/* static */
uint8_t GatePage::OnIncrement(uint8_t id, int8_t value) {
  // ENCA changes all step notes gate up/down
  if (id == ENCODER_A) {
    for (uint8_t n = 0; n < kNumSteps; n++) {
      uint8_t gate = dev.view_seq_step_gate(n);
      dev.set_view_seq_step_gate(n, Clamp(static_cast<int16_t>(gate) + value, 0, kNoteDurationCount - 1));
    }
    return 1;
  }
  // ENCx changes step note gate up/down
  if (id >= ENCODER_1 && id <= ENCODER_8) {
    uint8_t step = id - ENCODER_1;
    uint8_t gate = dev.view_seq_step_gate(step);
    dev.set_view_seq_step_gate(step, Clamp(static_cast<int16_t>(gate) + value, 0, kNoteDurationCount - 1));
    dev.SendViewSeqStep(step);
    return 1;
  }
  return 0;
}

/* static */
uint8_t GatePage::OnClick(uint8_t id, uint8_t value) {
  return 0;
}

/* static */
uint8_t GatePage::OnSwitch(uint8_t id, uint8_t value) {
  // Step switch toggles step legato
  if (id == SWITCH && value < kNumSteps) {
    uint8_t legato = dev.view_seq_step_legato(value);
    dev.set_view_seq_step_legato(value, !legato);
    return 1;
  }
  return 0;
}

/* static */
uint8_t GatePage::OnIdle() {
  return 0;
}

/* static */
uint8_t GatePage::UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  DrawSeparators();
  DrawStepIcons(line1);

  for (uint8_t n = 0; n < kNumSteps; n++) {
    uint8_t x = cell_pos(n);
    DrawSelStrN(&line1[x + 1], dev.view_seq_step_gate(n), midi_clock_ticks_per_note_str, kNoteDurationStrLen);
    if (dev.view_seq_step_legato(n)) {
      memcpy_P(&line2[x], PSTRN("lega"));
    }
  }

  return 1;
}

/* static */
uint8_t GatePage::UpdateLeds() {
  return 0;
}

} // namespace midialf
