// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// LFO configuration page.

#include "midialf/ui_pages/lfo_page.h"

namespace midialf {

/* <static> */
uint8_t LfoPage::lfo_;
LfoSettings LfoPage::prev_lfo_settings_;
uint8_t LfoPage::prev_resolution_;
/* </static> */

/* static */
const prog_EventHandlers LfoPage::event_handlers_ PROGMEM = {
  OnInit,
  OnQuit,
  OnIncrement,
  OnClick,
  OnSwitch,
  OnIdle,
  UpdateScreen,
  UpdateLeds,
};

/* static */
void LfoPage::OnInit(PageInfo* pageInfo, UiPageIndex prevPage) {
  if (prevPage <= LAST_PAGE) {
    lfo_ = prevPage < PAGE_LFO ? 0 : kNumLfos - 1;
  }

  InitPage();
}

/* static */
void LfoPage::OnQuit(UiPageIndex nextPage) {
}

#define LFO_SET_(name, Name) \
  lfo.set_##name(lfo_, Clamp(static_cast<int16_t>(lfo.name(lfo_)) + value, kMin##Name, kMax##Name))

#define LFO_SET2(name, Name) \
  lfo.set_##name(Clamp(static_cast<int16_t>(lfo.name()) + value, kMin##Name, kMax##Name))

/* static */
uint8_t LfoPage::OnIncrement(uint8_t id, int8_t value) {
  // ENCB selects LFO page
  if (id == ENCODER_B) {
    if (value > 0) {
      if (lfo_ == kNumLfos - 1) {
        return 0;
      } else {
        ++lfo_;
        InitPage();
      }
    } else
    if (value < 0) {
      if (lfo_ == 0) {
        return 0;
      } else {
        --lfo_;
        InitPage();
      }
    }
    return 1;
  }

  switch (id) {
    case ENCODER_1: LFO_SET_(waveform, LfoWaveform); break;
    case ENCODER_2: LFO_SET_(cc_number, LfoCCNumber); break;
    case ENCODER_3: LFO_SET_(amount, LfoAmount); break;
    case ENCODER_4: LFO_SET_(center, LfoCenter); break;
    case ENCODER_5: LFO_SET_(rate, LfoRate); break;
    case ENCODER_6: LFO_SET_(sync, LfoSync); break;
    case ENCODER_7: break;
    case ENCODER_8: LFO_SET2(resolution, LfoResolution); break;
  }

  return 1;
}

#undef LFO_SET_
#undef LFO_SET2

/* static */
uint8_t LfoPage::OnClick(uint8_t id, uint8_t value) {
  // ENCA shows command page
  if (id == ENCODER_A) {
    return 0;
  }
  // ENCB shows previous page
  if (id == ENCODER_B) {
    ui.ShowLastPage();
    return 1;
  }
  return 1;
}

#define TOGGLE_DEF_PREV_(name, Name) \
  lfo.set_##name(lfo_, lfo.name(lfo_) != kDef##Name ? kDef##Name : prev_lfo_settings_.name)

#define TOGGLE_DEF_PREV2(name, Name) \
  lfo.set_##name(lfo.name() != kDef##Name ? kDef##Name : prev_##name##_)

/* static */
uint8_t LfoPage::OnSwitch(uint8_t id, uint8_t value) {
  if (id != SWITCH)
    return 0;

  switch (value) {
    case SWITCH_1: TOGGLE_DEF_PREV_(waveform, LfoWaveform); break;
    case SWITCH_2: TOGGLE_DEF_PREV_(cc_number, LfoCCNumber); break;
    case SWITCH_3: TOGGLE_DEF_PREV_(amount, LfoAmount); break;
    case SWITCH_4: TOGGLE_DEF_PREV_(center, LfoCenter); break;
    case SWITCH_5: TOGGLE_DEF_PREV_(rate, LfoRate); break;
    case SWITCH_6: TOGGLE_DEF_PREV_(sync, LfoSync); break;
    case SWITCH_7: break;
    case SWITCH_8: TOGGLE_DEF_PREV2(resolution, LfoResolution); break;
  }
  return 1;
}

#undef TOGGLE_DEF_PREV_
#undef TOGGLE_DEF_PREV2

/* static */
uint8_t LfoPage::OnIdle() {
  return 0;
}

/* static */
uint8_t LfoPage::UpdateScreen() {
  DrawSeparators();

  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  DrawCells(line1, PSTR("LFO? CC AmntCentRateSync----Reso"));

  line1[cell_pos(0) + 3] = '1' + lfo_;

  // LFO waveshape
  DrawSelStr4(&line2[cell_pos(0)], lfo.waveform(lfo_), PSTR("none tri sqrramp sinsin2sin3sin5grg1grg2bat1bat2spk1spk2 lp1 lp2 rs1 rs2 s&h"));

  // LFO CC number
  UnsafeItoa(lfo.cc_number(lfo_), 3, &line2[cell_pos(1) + 1]);

  // LFO amount
  UnsafeItoa((int16_t)lfo.amount(lfo_) - 63, 3, &line2[cell_pos(2) + 1]);

  // LFO center
  UnsafeItoa(lfo.center(lfo_), 3, &line2[cell_pos(3) + 1]);

  // LFO rate
  DrawSelStrN(&line2[cell_pos(4) + 1], kNumLfoRates - 1 - lfo.rate(lfo_), 
    midi_clock_ticks_per_note_str + (kNoteDurationCount - kNumLfoRates) * kNoteDurationStrLen, kNoteDurationStrLen);
  
  // LFO sync
  DrawSelStr4(&line2[cell_pos(5)], lfo.sync(lfo_), PSTR("freestrt seqstep"));

  // LFO resolution (common for all LFOs)
  DrawSelStrN(&line2[cell_pos(7) + 1], lfo.resolution(), midi_clock_ticks_per_note_str, kNoteDurationStrLen);

  return 1;
}

/* static */
uint8_t LfoPage::UpdateLeds() {
  return 0;
}

/* static */
void LfoPage::InitPage() {
  lfo.get_settings(lfo_, prev_lfo_settings_);
  prev_resolution_ = lfo.resolution();
}

} // namespace midialf
