// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Load page class.

#include "midialf/ui_pages/load_page.h"
#include "midialf/ui_pages/save_page.h"
#include "midialf/storage.h"
#include "midialf/slot.h"

namespace midialf {

/* <static> */
uint8_t LoadPage::slot_;
uint8_t LoadPage::name_[kNameLength];
static const uint8_t kNameOffset = 5;
/* </static> */

/* static */
UiPageIndex LoadPage::prev_page_;

/* static */
const prog_EventHandlers LoadPage::event_handlers_ PROGMEM = {
  OnInit,
  OnQuit,
  OnIncrement,
  OnClick,
  OnSwitch,
  OnIdle,
  UpdateScreen,
  UpdateLeds,
};

/* static */
void LoadPage::OnInit(PageInfo* pageInfo, UiPageIndex prevPage) {
  if (prevPage == PAGE_SAVE) prev_page_ = SavePage::prev_page(); else
  if (prevPage != PAGE_LOAD) prev_page_ = prevPage;
  SetSlot(dev.slot());
}

/* static */
void LoadPage::OnQuit(UiPageIndex nextPage) {
}

/* static */
uint8_t LoadPage::OnIncrement(uint8_t id, int8_t value) {
  // ENCB changes slot number
  if (id == ENCODER_B) {
    uint8_t slot = Clamp((int16_t)slot_ + value, 0, storage.num_slots() - 1);
    SetSlot(slot);
  }
  return 1;
}

/* static */
uint8_t LoadPage::OnClick(uint8_t id, uint8_t value) {
  if (value == CLICK || value == DBLCLICK) {
    // ENCB loads slot and cancels mode
    if (id == ENCODER_B) {
      LoadSlot();
      Ui::ShowPage(prev_page_);
    } else
    // ENC8 cancels mode
    if (id == ENCODER_8) {
      Ui::ShowPage(prev_page_);
    }
    return 1;
  }

  return 0;
}

/* static */
uint8_t LoadPage::OnSwitch(uint8_t id, uint8_t value) {
  if (id == SWITCH) {
    return 1;
  }

  if (id == SIDE_SWITCH) {
    switch (value) {
      case SIDE_SWITCH_LOAD:
        LoadSlot();
        Ui::ShowPage(prev_page_);
        return 1;
    }
  }
  return 0;
}

/* static */
uint8_t LoadPage::OnIdle() {
  return 0;
}

/* static */
uint8_t LoadPage::UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  static const prog_char cmdLoad[] PROGMEM = "[Load]";
  static const prog_char cmdExit[] PROGMEM = "[Exit]";

  memcpy_P(&line1[0], PSTRN("Load from"));
  memcpy_P(&line1[kLcdWidth - lengof(cmdLoad)], cmdLoad, lengof(cmdLoad));
  memcpy_P(&line2[kLcdWidth - lengof(cmdExit)], cmdExit, lengof(cmdExit));

  Ui::PrintNNN(&line2[1], 1 + slot_);
  memcpy(&line2[kNameOffset], name_, kNameLength);

  return 1;
}

/* static */
uint8_t LoadPage::UpdateLeds() {
  return 0;
}

/* static */
void LoadPage::SetSlot(uint8_t slot) {
  slot_ = slot;
  Slot::ReadName(slot, name_);
}

/* static */
void LoadPage::LoadSlot() {
  dev.LoadSlot(slot_);
}

} // namespace midialf
