// Copyright 2012 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Save page class.

#include "midialf/ui_pages/save_page.h"
#include "midialf/ui_pages/load_page.h"
#include "midialf/storage.h"
#include "midialf/slot.h"

namespace midialf {

/* <static> */
uint8_t SavePage::slot_;
uint8_t SavePage::name_[kNameLength];
uint8_t SavePage::name_edited_;
uint8_t SavePage::edit_offset_;
UiPageIndex SavePage::prev_page_;
static const uint8_t kNameOffset = 5;
/* </static> */

/* static */
const prog_EventHandlers SavePage::event_handlers_ PROGMEM = {
  OnInit,
  OnQuit,
  OnIncrement,
  OnClick,
  OnSwitch,
  OnIdle,
  UpdateScreen,
  UpdateLeds,
};

/* static */
void SavePage::OnInit(PageInfo* pageInfo, UiPageIndex prevPage) {
  if (prevPage == PAGE_LOAD) prev_page_ = LoadPage::prev_page(); else
  if (prevPage != PAGE_SAVE) prev_page_ = prevPage;
  SetSlot(dev.slot());
}

/* static */
void SavePage::OnQuit(UiPageIndex nextPage) {
  display.set_cursor_position(kLcdNoCursor);
}

/* static */
uint8_t SavePage::OnIncrement(uint8_t id, int8_t value) {
  // ENCB changes slot number
  if (id == ENCODER_B) {
    uint8_t slot = Clamp((int16_t)slot_ + value, 0, storage.num_slots() - 1);
    SetSlot(slot);
  } else {
    switch (id) {
      // ENC1 increments/decrements character at cursor
      case ENCODER_1: 
        if (value < 0) {
          if (name_[edit_offset_] > kMinNameChar) {
            name_[edit_offset_]--;
            name_edited_ = 1;
          }
        } else 
        if (value > 0) {
          if (name_[edit_offset_] < kMaxNameChar) {
            name_[edit_offset_]++;
            name_edited_ = 1;
          }
        }
        break;
      // ENC2 shifts cursor left/right
      case ENCODER_2: 
        edit_offset_ = Clamp((int16_t)edit_offset_ + value, 0, kNameLength - 1);
        break;
    }
  }
  return 1;
}

/* static */
uint8_t SavePage::OnClick(uint8_t id, uint8_t value) {
  if (value != CLICK)
    return 0;

  // ENCB saves slot and cancels mode
  if (id == ENCODER_B) {
    SaveSlot();
    Ui::ShowPage(prev_page_);
  } else
  // ENC1 decrements offset
  if (id == ENCODER_1) {
    edit_offset_ = Clamp((int16_t)edit_offset_  - 1, 0, kNameLength - 1);
  } else
  // ENC2 increments offset
  if (id == ENCODER_2) {
    edit_offset_ = Clamp((int16_t)edit_offset_  + 1, 0, kNameLength - 1);
  } else
  // ENC8 cancels mode
  if (id == ENCODER_8) {
    Ui::ShowPage(prev_page_);
  }
  return 1;
}

/* static */
uint8_t SavePage::OnSwitch(uint8_t id, uint8_t value) {
  if (id == SWITCH) {
    uint8_t chr;
    switch (value) {
      case SWITCH_1: // Insert at cursor
        { uint8_t cch = kNameLength - edit_offset_ - 1;
          memmove(&name_[edit_offset_ + 1], &name_[edit_offset_], cch);
          name_[edit_offset_] = ' ';
          name_edited_ = 1;
        }
        break;
      case SWITCH_2: // Delete at cursor
        { uint8_t cch = kNameLength - edit_offset_ - 1;
          memmove(&name_[edit_offset_], &name_[edit_offset_ + 1], cch);
          name_[kNameLength - 1] = ' ';
          name_edited_ = 1;
        }
        break;
    }
    return 1;
  }

  if (id == SIDE_SWITCH) {
    switch (value) {
      case SIDE_SWITCH_SAVE:
        SaveSlot();
        Ui::ShowPage(prev_page_);
        return 1;
    }
  }
  return 0;
}

/* static */
uint8_t SavePage::OnIdle() {
  return 0;
}

/* static */
uint8_t SavePage::UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  static const prog_char prompt[] PROGMEM = "Save ";
  static const prog_char prompt_end[] PROGMEM = "to ";
  static const prog_char cmdSave[] PROGMEM = "[Save]";
  static const prog_char cmdExit[] PROGMEM = "[Exit]";

  memcpy_P(&line1[0], prompt, lengof(prompt));
  dev.get_name((uint8_t*)&line1[lengof(prompt)]);
  uint8_t x = FindLastSpace(&line1[lengof(prompt)], kNameLength);
  if (lengof(prompt) + x + lengof(prompt_end) < kLcdWidth - lengof(cmdSave)) x++;
  memcpy_P(&line1[lengof(prompt) + x], PSTRN("to "));
  memcpy_P(&line1[kLcdWidth - lengof(cmdSave)], cmdSave, lengof(cmdSave));

  Ui::PrintNNN(&line2[1], 1 + slot_);
  memcpy(&line2[kNameOffset], name_, kNameLength);
  memcpy_P(&line2[kLcdWidth - lengof(cmdExit)], cmdExit, lengof(cmdExit));

  display.set_cursor_position(kLcdWidth + kNameOffset + edit_offset_);

  return 1;
}

/* static */
uint8_t SavePage::UpdateLeds() {
  return 0;
}

/* static */
void SavePage::SetSlot(uint8_t slot) {
  slot_ = slot;
  Slot::ReadName(slot, name_);
  name_edited_ = 0;
  edit_offset_ = 0;
}

/* static */
void SavePage::SaveSlot() {
  if (name_edited_) {
    dev.set_name(name_);
    name_edited_ = 0;
  }

  dev.SaveSlot(slot_);
}

} // namespace midialf
