// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Song page.

#include "midialf/ui_pages/song_page.h"

namespace midialf {

/* <static> */
uint8_t SongPage::step_;
uint8_t SongPage::learning_root_note_;
/* </static> */

/* static */
const prog_EventHandlers SongPage::event_handlers_ PROGMEM = {
  OnInit,
  OnQuit,
  OnIncrement,
  OnClick,
  OnSwitch,
  OnIdle,
  UpdateScreen,
  UpdateLeds,
};

/* static */
void SongPage::OnInit(PageInfo* pageInfo, UiPageIndex prevPage) {
  learning_root_note_ = 0;
  dev.set_note_callback(NoteCallback);
}

/* static */
void SongPage::OnQuit(UiPageIndex nextPage) {
  dev.set_note_callback(0);
}

/* static */
uint8_t SongPage::OnIncrement(uint8_t id, int8_t value) {
  // ENCA selects song step
  if (id == ENCODER_A) {
    if (value > 0) {
      if (step_ < kNumSteps - 1) {
        ++step_;
      }
    } else
    if (value < 0) {
      if (step_ > 0) {
        --step_;
      }
    }
    return 1;
  }
  // ENCx changes song step play count or root note
  if (id >= ENCODER_1 && id <= ENCODER_8) {
    step_ = id - ENCODER_1;
    if (learning_root_note_) {
      value = Ui::FixOctaveIncrement(value);
      dev.set_song_step_root_note(step_, Clamp(static_cast<int16_t>(dev.song_step_root_note(step_)) + value, kMinSongRootNote, kMaxSongRootNote));
      if (dev.running() && dev.song_step() == step_) {
        dev.set_transpose_note(dev.song_step_root_note(step_));
      }
    } else {
      dev.set_song_step_play_count(step_, Clamp(static_cast<int16_t>(dev.song_step_play_count(step_)) + value, kMinSongPlayCount, kMaxSongPlayCount));
    }
    return 1;
  }
  return 0;
}

/* static */
uint8_t SongPage::OnClick(uint8_t id, uint8_t value) {
  // ENCA toggles song mode
  if (id == ENCODER_A) {
    dev.set_song_mode_active(!dev.song_mode_active());
    return 1;
  }
  // ENCB shows previous page
  if (id == ENCODER_B) {
    ui.ShowLastPage();
    return 1;
  }
  // ENCx sepects song step
  if (id >= ENCODER_1 && id <= ENCODER_8) {
    step_ = id - ENCODER_1;
    if (dev.running()) {
      dev.SetSongStep(step_);
    }
    return 1;
  }
  return 0;
}

/* static */
uint8_t SongPage::OnSwitch(uint8_t id, uint8_t value) {
  // Seq switch selects step sequence
  if (id == SIDE_SWITCH && value >= SIDE_SWITCH_SEQA && value <= SIDE_SWITCH_SEQD) {
    uint8_t seq = value - SIDE_SWITCH_SEQA;
    dev.set_song_step_seq(step_, seq);
    if (dev.running() && dev.song_step() == step_) {
      dev.set_play_seq(dev.song_step_seq(step_));
    }
    return 1;
  }
  // Step switch toggles learning root note mode
  if (id == SWITCH && value < kNumSteps) {
    if (step_ != value) {
      step_ = value;
      learning_root_note_ = 1;
    } else {
      learning_root_note_ = !learning_root_note_;
    }
    return 1;
  }
  return 0;
}

/* static */
uint8_t SongPage::OnIdle() {
  return 0;
}

/* static */
uint8_t SongPage::UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  DrawSeparators();

  for (uint8_t n = 0; n < kNumSteps; n++) {
    uint8_t x = cell_pos(n);
    line1[x] = 'A' + dev.song_step_seq(n);
    line1[x + 1] = ':';
    uint8_t step_play_count = dev.song_step_play_count(n);
    if (step_play_count) {
      UnsafeItoa(step_play_count, 2, &line1[x + 2]);
    } else
      Draw2Dashes(&line1[x + 2]);

    Ui::PrintNote(&line2[x + 1], dev.song_step_root_note(n));

    if (learning_root_note_ && n == step_) {
      DrawBrackets(&line2[x], 3);
    }
  }

  return 1;
}

/* static */
uint8_t SongPage::UpdateLeds() {
  // Update seq LEDs
  for (uint8_t n = 0; n < kNumSeqs; n++) {
    uint8_t intensity = 0;

    // Currently selected step sequence is fully lit, others are dim
    if (n == dev.song_step_seq(step_)) {
      intensity = 0xf;
    } else {
      intensity = 1;
    }

    // The currently playing sequence is always blinking
    if (dev.running() && n == dev.play_seq()) {
      uint8_t half_step = dev.clock_prescaler() / 2;
      if (intensity == 0xf) {
        intensity = ui.tick_count() < half_step ? 0xf : 0;
      } else {
        intensity = ui.tick_count() < half_step ? 0x5 : 0;
      }
    }

    leds.set_pixel(LED_SEQA + n, intensity);
  }

  // Update step LEDs
  for (uint8_t n = 0; n < kNumSteps; n++) {
    uint8_t intensity = 0;

    // Currently selected song step is fully lit, others dimmed.
    if (n == step_) {
      intensity = 0xf;
    } else {
      intensity = 1;
    }

    // Currently playing song step is half lit and blinking.
    if (dev.running() && n == dev.song_step()) {
      uint8_t half_step = dev.clock_prescaler() / 2;
      if (intensity == 0xf) {
        intensity = ui.tick_count() < half_step ? 0xf : 0;
      } else {
        intensity = ui.tick_count() < half_step ? 0x5 : 0;
      }
    }

    // Currently playing sequence step is half lit.
    if (dev.running && n == dev.step() && intensity == 1) {
      intensity = 5;
    }

    leds.set_pixel(LED_1 + n, intensity);
  }

  ui.HandleUpdateRightSideLeds();
  return 1;
}

/* static */
uint8_t SongPage::NoteCallback(uint8_t channel, uint8_t note, uint8_t velocity) {
  if (learning_root_note_ && channel == dev.channel()) {
    dev.set_song_step_root_note(step_, note);
    ui.request_redraw();
    learning_root_note_ = 0;
  }

  return 0;
}


} // namespace midialf
