// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Step configuration pages.

#include "midialf/ui_pages/step_page.h"

#include <stdlib.h>

namespace midialf {

/* <static> */
UiPageIndex StepPage::prev_page_;
uint8_t StepPage::page_;
uint8_t StepPage::step_;
StepPage::Prev* StepPage::prev_;
uint32_t StepPage::scale_selected_tick_;
/* </static> */

static const uint16_t kShowScaleMilliseconds = 2000;

/* static */
const prog_EventHandlers StepPage::event_handlers_ PROGMEM = {
  OnInit,
  OnQuit,
  OnIncrement,
  OnClick,
  OnSwitch,
  OnIdle,
  UpdateScreen,
  UpdateLeds,
};

/* static */
void StepPage::OnInit(PageInfo* pageInfo, UiPageIndex prevPage) {
  if (prevPage <= LAST_PAGE) {
    prev_page_ = prevPage;
  }

  prev_ = (Prev*)malloc(sizeof(Prev));

  InitPage();
}

/* static */
void StepPage::OnQuit(UiPageIndex nextPage) {
  free(prev_);
}

/* static */
uint8_t StepPage::OnIncrement(uint8_t id, int8_t value) {
  // ENCA selects step
  if (id == ENCODER_A) {
    if (value > 0) {
      if (step_ < kNumSteps - 1) {
        ++step_;
        InitPage();
      }
    } else
    if (value < 0) {
      if (step_ > 0) {
        --step_;
        InitPage();
      }
    }
    return 1;
  }

  // ENCB selects step configuration page
  if (id == ENCODER_B) {
    if (value > 0) {
      if (page_ < kNumPages - 1) {
        ++page_;
        InitPage();
      }
    } else
    if (value < 0) {
      if (page_ > 0) {
        --page_;
        InitPage();
      }
    }
    return 1;
  }

  switch (page_) {
    case 0: return HandlePage1_OnIncrement(id, value);
    case 1: return HandlePage2_OnIncrement(id, value);
    case 2: return HandlePage3_OnIncrement(id, value);
  }
  return 1;
}

#define DEV_SET(name, Name) \
  dev.set_view_seq_step_##name(step_, Clamp(static_cast<int16_t>(dev.view_seq_step_##name(step_)) + value, kMin##Name, kMax##Name))

#define DEV_SET_FLAG(name, value) \
  dev.set_view_seq_step_##name(step_, value)

#define DEV_HAS_FLAG(name) \
  dev.view_seq_step_##name(step_)

#define DEV_SET_WITH_FLAG(name, Name) \
  if (DEV_HAS_FLAG(has_##name)) { \
    if (value < 0 && dev.view_seq_step_##name(step_) == kMin##Name) { \
      DEV_SET_FLAG(has_##name, 0); \
    } else \
      DEV_SET(name, Name); \
  } else { \
    if (!(value < 0 && dev.view_seq_step_##name(step_) == kMin##Name)) { \
      DEV_SET_FLAG(has_##name, 1); \
    } \
  }

#define DEV_SET_WITH_FLAG2(name, Name, Flag) \
  if (DEV_HAS_FLAG(Flag)) { \
    if (value < 0 && dev.view_seq_step_##name(step_) == kMin##Name) { \
      DEV_SET_FLAG(Flag, 0); \
    } else \
      DEV_SET(name, Name); \
  } else { \
    if (!(value < 0 && dev.view_seq_step_##name(step_) == kMin##Name)) { \
      DEV_SET_FLAG(Flag, 1); \
    } \
  }

/* static */
uint8_t StepPage::HandlePage1_OnIncrement(uint8_t id, int8_t value) {
  switch (id) {
    case ENCODER_1: value = Ui::FixOctaveIncrement(value); DEV_SET(note, StepNote); dev.SendViewSeqStep(step_); break;
    case ENCODER_2: DEV_SET(velocity, StepVelocity); dev.SendViewSeqStep(step_); break;
    case ENCODER_3: DEV_SET(gate, StepGate); dev.SendViewSeqStep(step_); break;
    case ENCODER_4: DEV_SET_FLAG(mute, value > 0); break;
    case ENCODER_5: DEV_SET_FLAG(skip, value > 0); break;
    case ENCODER_6: DEV_SET_FLAG(legato, value > 0); break;
    case ENCODER_7: DEV_SET_WITH_FLAG2(cc1, StepCC, send_cc1); dev.SendViewSeqStep(step_); break;
    case ENCODER_8: DEV_SET_WITH_FLAG2(cc2, StepCC, send_cc2); dev.SendViewSeqStep(step_); break;
  }
  return 1;
}

/* static */
uint8_t StepPage::HandlePage2_OnIncrement(uint8_t id, int8_t value) {
  switch (id) {
    case ENCODER_1: DEV_SET_WITH_FLAG(condition, StepCondition); break;
    case ENCODER_2: DEV_SET_WITH_FLAG(cc1_condition, StepCC1Condition); break;
    case ENCODER_3: DEV_SET_WITH_FLAG(cc2_condition, StepCC2Condition); break;
    case ENCODER_4: DEV_SET(randomize_note_scale, StepRandomizeNoteScale); scale_selected_tick_ = milliseconds(); break;
    case ENCODER_5: value = Ui::FixOctaveIncrement(value); DEV_SET(randomize_note, StepRandomizeNote); break;
    case ENCODER_6: DEV_SET(randomize_velocity, StepRandomizeVelocity); break;
    case ENCODER_7: DEV_SET(randomize_cc1, StepRandomizeCC1); break;
    case ENCODER_8: DEV_SET(randomize_cc2, StepRandomizeCC2); break;
  }

  if (id > ENCODER_4 && scale_selected_tick_ > 0) {
    scale_selected_tick_ = 0;
  }

  return 1;
}

/* static */
uint8_t StepPage::HandlePage3_OnIncrement(uint8_t id, int8_t value) {
  switch (id) {
    case ENCODER_1: DEV_SET_WITH_FLAG(retrigger, StepRetrigger); break;
    case ENCODER_2: DEV_SET(retrigger_velocity, StepRetriggerVelocity); break;
    case ENCODER_3: value = Ui::FixOctaveIncrement(value); DEV_SET(retrigger_transposition, StepRetriggerTransposition); break;
    case ENCODER_4: DEV_SET(retrigger_transposition_scale, StepRetriggerTranspositionScale); scale_selected_tick_ = milliseconds(); break;
    case ENCODER_5: DEV_SET_WITH_FLAG(retrigger_condition, StepRetriggerCondition); break;
    case ENCODER_6: break;
    case ENCODER_7: break;
    case ENCODER_8: break;
  }

  if (id > ENCODER_4 && scale_selected_tick_ > 0) {
    scale_selected_tick_ = 0;
  }

  return 1;
}

#undef DEV_SET
#undef DEV_SET_FLAG

/* static */
uint8_t StepPage::OnClick(uint8_t id, uint8_t value) {
  // ENCA shows command page
  if (id == ENCODER_A) {
    return 0;
  }
  // ENCB shows previous page
  if (id == ENCODER_B) {
    ui.ShowPage(prev_page_);
    return 1;
  }
  // ENCx click selects a step or closes the selected step
  if (id >= ENCODER_1 && id <= ENCODER_8) {
    uint8_t step = id - ENCODER_1;
    if (step == step_) {
      ui.ShowPage(prev_page_);
    } else {
      step_ = step;
      InitPage();
    }
    return 1;
  }
  return 1;
}

/* static */
uint8_t StepPage::OnSwitch(uint8_t id, uint8_t value) {
  if (id != SWITCH)
    return 0;

  switch (page_) {
    case 0: return HandlePage1_OnSwitch(id, value);
    case 1: return HandlePage2_OnSwitch(id, value);
    case 2: return HandlePage3_OnSwitch(id, value);
  }
  return 1;
}

#define TOGGLE_DEF_PREV(name, Name) \
  dev.set_view_seq_step_##name(step_, dev.view_seq_step_##name(step_) != kDef##Name ? kDef##Name : prev_->name)

#define TOGGLE_DEV_FLAG(name) \
  dev.set_view_seq_step_##name(step_, dev.view_seq_step_##name(step_) ? 0 : 1)

/* static */
uint8_t StepPage::HandlePage1_OnSwitch(uint8_t id, uint8_t value) {
  switch (value) {
    case SWITCH_1: TOGGLE_DEF_PREV(note, StepNote); break;
    case SWITCH_2: TOGGLE_DEF_PREV(velocity, StepVelocity); break;
    case SWITCH_3: TOGGLE_DEF_PREV(gate, StepGate); break;
    case SWITCH_4: TOGGLE_DEV_FLAG(mute); break;
    case SWITCH_5: TOGGLE_DEV_FLAG(skip); break;
    case SWITCH_6: TOGGLE_DEV_FLAG(legato); break;
    case SWITCH_7: TOGGLE_DEV_FLAG(send_cc1); break;
    case SWITCH_8: TOGGLE_DEV_FLAG(send_cc2); break;
  }
  return 1;
}

/* static */
uint8_t StepPage::HandlePage2_OnSwitch(uint8_t id, uint8_t value) {
  switch (value) {
    case SWITCH_1: TOGGLE_DEV_FLAG(has_condition); break;
    case SWITCH_2: TOGGLE_DEV_FLAG(has_cc1_condition); break;
    case SWITCH_3: TOGGLE_DEV_FLAG(has_cc2_condition); break;
    case SWITCH_4: TOGGLE_DEF_PREV(randomize_note_scale, StepRandomizeNoteScale); scale_selected_tick_ = milliseconds(); break;
    case SWITCH_5: TOGGLE_DEF_PREV(randomize_note, StepRandomizeNote); break;
    case SWITCH_6: TOGGLE_DEF_PREV(randomize_velocity, StepRandomizeVelocity); break;
    case SWITCH_7: TOGGLE_DEF_PREV(randomize_cc1, StepRandomizeCC1); break;
    case SWITCH_8: TOGGLE_DEF_PREV(randomize_cc2, StepRandomizeCC2); break;
  }
  return 1;
}

/* static */
uint8_t StepPage::HandlePage3_OnSwitch(uint8_t id, uint8_t value) {
  switch (value) {
    case SWITCH_1: TOGGLE_DEV_FLAG(has_retrigger); break;
    case SWITCH_2: TOGGLE_DEF_PREV(retrigger_velocity, StepRetriggerVelocity); break;
    case SWITCH_3: TOGGLE_DEF_PREV(retrigger_transposition, StepRetriggerTransposition); break;
    case SWITCH_4: TOGGLE_DEF_PREV(retrigger_transposition_scale, StepRetriggerTranspositionScale); scale_selected_tick_ = milliseconds(); break;
    case SWITCH_5: TOGGLE_DEV_FLAG(has_retrigger_condition); break;
    case SWITCH_6: break;
    case SWITCH_7: break;
    case SWITCH_8: break;
  }

  if (value > SWITCH_4 && scale_selected_tick_ > 0) {
    scale_selected_tick_ = 0;
  }

  return 1;
}

#undef TOGGLE_DEF_PREV
#undef TOGGLE_DEV_FLAG

/* static */
uint8_t StepPage::OnIdle() {
  return 0;
}

/* static */
uint8_t StepPage::UpdateScreen() {
  DrawSeparators();

  switch (page_) {
    case 0: HandlePage1_UpdateScreen(); break;
    case 1: HandlePage2_UpdateScreen(); break;
    case 2: HandlePage3_UpdateScreen(); break;
  }
  return 1;
}

/* static */
void StepPage::HandlePage1_UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  DrawCells(line1, PSTR("NoteVeloGateMuteSkipLega CC1 CC2"));

  // Step note/velocity/gate
  Ui::PrintNote(&line2[cell_pos(0) + 1], dev.view_seq_step_note(step_));
  UnsafeItoa(dev.view_seq_step_velocity(step_), 3, &line2[cell_pos(1) + 1]);
  DrawSelStrN(&line2[cell_pos(2) + 1], dev.view_seq_step_gate(step_), midi_clock_ticks_per_note_str, kNoteDurationStrLen);

  // Step mute/skip/legato
  DrawSelStr4(&line2[cell_pos(3)], dev.view_seq_step_mute(step_), pchOffOn);
  DrawSelStr4(&line2[cell_pos(4)], dev.view_seq_step_skip(step_), pchOffOn);
  DrawSelStr4(&line2[cell_pos(5)], dev.view_seq_step_legato(step_), pchOffOn);

  // Step CC1
  if (dev.view_seq_step_send_cc1(step_)) {
    UnsafeItoa(dev.view_seq_step_cc1(step_), 3, &line2[cell_pos(6) + 1]);
  } else
    Draw2Dashes(&line2[cell_pos(6) + 1]);

  // Step CC2
  if (dev.view_seq_step_send_cc2(step_)) {
    UnsafeItoa(dev.view_seq_step_cc2(step_), 3, &line2[cell_pos(7) + 1]);
  } else
    Draw2Dashes(&line2[cell_pos(7) + 1]);
}

/* static */
void StepPage::HandlePage2_UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  DrawCells(line1, PSTR("CondCCC1CCC2ScleRNotRVelRCC1RCC2"));

  // Step condition
  if (dev.view_seq_step_has_condition(step_)) {
    Cond::get_name(dev.view_seq_step_condition(step_), &line2[cell_pos(0)]);
  } else
    Draw2Dashes(&line2[cell_pos(0) + 1]);

  // Step cc1 condition
  if (dev.view_seq_step_has_cc1_condition(step_)) {
    Cond::get_name(dev.view_seq_step_cc1_condition(step_), &line2[cell_pos(1)]);
  } else
    Draw2Dashes(&line2[cell_pos(1) + 1]);

  // Step cc2 condition
  if (dev.view_seq_step_has_cc2_condition(step_)) {
    Cond::get_name(dev.view_seq_step_cc2_condition(step_), &line2[cell_pos(2)]);
  } else
    Draw2Dashes(&line2[cell_pos(2) + 1]);

  // Show randomize note scale
  uint8_t randomize_scale = dev.view_seq_step_randomize_note_scale(step_);
  uint8_t x_scale = cell_pos(3) + 1;
  if (randomize_scale > 0) {
    x_scale+= UnsafeItoaLen(randomize_scale, 3, &line2[x_scale]);
  } else
    memcpy_P(&line2[cell_pos(3)], PSTRN("none"));

  // Step randomize note, velocity, CC1/2 values
  UnsafeItoa(dev.view_seq_step_randomize_note(step_), 3, &line2[cell_pos(4) + 1]);
  UnsafeItoa(dev.view_seq_step_randomize_velocity(step_), 3, &line2[cell_pos(5) + 1]);
  UnsafeItoa(dev.view_seq_step_randomize_cc1(step_), 3, &line2[cell_pos(6) + 1]);
  UnsafeItoa(dev.view_seq_step_randomize_cc2(step_), 3, &line2[cell_pos(7) + 1]);

  // Show randomize note scale name
  if (randomize_scale > 0 && scale_selected_tick_ > 0) {
    line2[x_scale] = ' ';
    Scale::GetScaleName(randomize_scale - 1, (uint8_t*)&line2[x_scale + 1]);
  }
}

/* static */
void StepPage::HandlePage3_UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  DrawCells(line1, PSTR("TrigVeloTranScleCond------------"));

  // Step retrigger
  if (dev.view_seq_step_has_retrigger(step_)) {
    UnsafeItoa(dev.view_seq_step_retrigger(step_), 2, &line2[cell_pos(0) + 1]);
  } else
    Draw2Dashes(&line2[cell_pos(0) + 1]);

  // Step retrigger velocity and transposition
  UnsafeItoa((int8_t)dev.view_seq_step_retrigger_velocity(step_) - kDefStepRetriggerVelocity, 3, &line2[cell_pos(1) + 1]);
  UnsafeItoa((int8_t)dev.view_seq_step_retrigger_transposition(step_) - kDefStepRetriggerTransposition, 3, &line2[cell_pos(2) + 1]);

  uint8_t transposition_scale = dev.view_seq_step_retrigger_transposition_scale(step_);
  uint8_t x_scale = cell_pos(3) + 1;
  if (transposition_scale > 0) {
    x_scale+= UnsafeItoaLen(transposition_scale, 3, &line2[x_scale]);
  } else
    memcpy_P(&line2[cell_pos(3)], PSTRN("none"));

  // Step retrigger condition
  if (dev.view_seq_step_has_retrigger_condition(step_)) {
    Cond::get_name(dev.view_seq_step_retrigger_condition(step_), &line2[cell_pos(4)]);
  } else
    Draw2Dashes(&line2[cell_pos(4) + 1]);

  // Show transposition scale name
  if (transposition_scale > 0 && scale_selected_tick_ > 0) {
    line2[x_scale] = ' ';
    Scale::GetScaleName(transposition_scale - 1, (uint8_t*)&line2[x_scale + 1]);
  }
}

/* static */
uint8_t StepPage::UpdateLeds() {
  for (uint8_t n = 0; n < kNumSteps; n++) {
    uint8_t intensity = 0;

    // Currently selected sequence step is fully lit, others dimmed.
    if (n == step_) {
      intensity = 0xf;
    } else {
      intensity = 1;
    }

    // Currently playing sequence step is half lit.
    if (dev.running && n == dev.step() && intensity == 1) {
      intensity = 5;
    }

    leds.set_pixel(LED_1 + n, intensity);
  }

  ui.HandleUpdateLeftSideLeds();
  ui.HandleUpdateRightSideLeds();

  // Stop showing transposition scale
  if (scale_selected_tick_ && (milliseconds() - scale_selected_tick_) > kShowScaleMilliseconds) {
    scale_selected_tick_ = 0;
    ui.request_redraw();
  }

  return 1;
}

/* static */
void StepPage::InitPage() {
  prev_->note = dev.view_seq_step_note(step_);
  prev_->velocity = dev.view_seq_step_velocity(step_);
  prev_->gate = dev.view_seq_step_gate(step_);
  prev_->randomize_note = dev.view_seq_step_randomize_note(step_);
  prev_->randomize_note_scale = dev.view_seq_step_randomize_note_scale(step_);
  prev_->randomize_velocity = dev.view_seq_step_randomize_velocity(step_);
  prev_->randomize_cc1 = dev.view_seq_step_randomize_cc1(step_);
  prev_->randomize_cc2 = dev.view_seq_step_randomize_cc2(step_);
  prev_->retrigger = dev.view_seq_step_retrigger(step_);
  prev_->retrigger_velocity = dev.view_seq_step_retrigger_velocity(step_);
  prev_->retrigger_transposition = dev.view_seq_step_retrigger_transposition(step_);
  prev_->retrigger_transposition_scale = dev.view_seq_step_retrigger_transposition_scale(step_);

  scale_selected_tick_ = 0;
}

} // namespace midialf
