// Copyright 2018 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// UI page to report sysex progress.

#include "midialf/ui_pages/sysex_page.h"
#include "midialf/sysex.h"

namespace midialf {

/* <static> */
UiPageIndex SysexPage::prev_page_;
/* </static> */

/* static */
const prog_EventHandlers SysexPage::event_handlers_ PROGMEM = {
  OnInit,
  OnQuit,
  OnIncrement,
  OnClick,
  OnSwitch,
  OnIdle,
  UpdateScreen,
  UpdateLeds,
};

/* static */
void SysexPage::OnInit(PageInfo* pageInfo, UiPageIndex prevPage) {
  dev.Stop();
  dev.StopRecording();

  if (prevPage != PAGE_SYSEX) {
    prev_page_ = prevPage;
  }
}

/* static */
void SysexPage::OnQuit(UiPageIndex nextPage) {
  sysex.FreeBuffer();
}

/* static */
uint8_t SysexPage::OnIncrement(uint8_t id, int8_t value) {
  return 1;
}

/* static */
uint8_t SysexPage::OnClick(uint8_t id, uint8_t value) {
  // ENCB cancel sysex operation or activates previous page
  if (id == ENCODER_B) {
    if (sysex.IsSendingSysex() || sysex.IsReceivingSysex()) {
      sysex.Cancel();
    } else {
      ui.ShowPage(prev_page_);
    }
  }
  return 1;
}

/* static */
uint8_t SysexPage::OnSwitch(uint8_t id, uint8_t value) {
  return 1;
}

/* static */
uint8_t SysexPage::OnIdle() {
  return 0;
}

/* static */
uint8_t SysexPage::UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  switch (sysex.state()) {
    case SENDING_SYSEX:
      DrawSendingSysex(line1);
      goto Cancel;
    case SENDING_SYSEX_DONE:
    case SENDING_SYSEX_CANCELED:
      DrawSendingSysex(line1);
      goto Done;
    case RECEIVING_HEADER:
    case RECEIVING_COMMAND:
    case RECEIVING_DATA:
      DrawReceivingSysex(line1);
Cancel: static const prog_char cmdCancel[] PROGMEM = "[Cancel]";
      memcpy_P(&line1[kLcdWidth - lengof(cmdCancel)], cmdCancel, lengof(cmdCancel));
      break;
    case RECEPTION_OK:
      DrawReceivingSysex(line1);
      memcpy_P(&line2[0], PSTRN("OK"));
      goto Done;
    case RECEPTION_ERROR:
      DrawReceivingSysex(line1);
      static const prog_char failure[] PROGMEM = "FAILURE";
      memcpy_P(&line2[0], failure, lengof(failure));
      { uint8_t x = lengof(failure) + 1;
        x+= UnsafeItoaLen(sysex.error_code(), 3, &line2[x]) + 1;
        x+= UnsafeItoaLen(sysex.bytes_received(), 5, &line2[x]) + 1;
      }
      goto Done;
    default:
      memcpy_P(&line1[0], PSTRN("???"));
      static const prog_char cmdDone[] PROGMEM = "[Done]";
Done: memcpy_P(&line1[kLcdWidth - lengof(cmdDone)], cmdDone, lengof(cmdDone));
      break;
  }

  return 1;
}

/* static */
uint8_t SysexPage::UpdateLeds() {
  return 0;
}

/* static */
void SysexPage::DrawSendingSysex(char* line) {
  static const prog_char prompt[] PROGMEM = "Sending sysex";
  memcpy_P(&line[0], prompt, lengof(prompt));
  DrawSysexInfo(line + lengof(prompt));
}

/* static */
void SysexPage::DrawReceivingSysex(char* line) {
  static const prog_char prompt[] PROGMEM = "Receiving sysex";
  memcpy_P(&line[0], prompt, lengof(prompt));
  DrawSysexInfo(line + lengof(prompt));
}

/* static */
void SysexPage::DrawSysexInfo(char* line) {
  if (sysex.slot() == kCurSlot) {
    memcpy_P(line, PSTRN("..."));
  } else {
    static const prog_char slot[] PROGMEM = " slot ";
    memcpy_P(&line[0], slot, lengof(slot));
    UnsafeItoa(1 + sysex.slot(), 3, &line[lengof(slot)]);
  }
}

} // namespace midialf
