// Copyright 2014 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// Based on Ambika code by Olivier Gillet (ol.gillet@gmail.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// Base UI page class.

#include "midialf/ui_pages/ui_page.h"

#include "avrlib/string.h"

#include "midialf/midialf.h"

namespace midialf {

/* <static> */
// Cannot use resource strings specification for these because it strips leading 
// and trailing spaces.
const prog_char UiPage::pchOffOn[] PROGMEM = " off on ";
/* </static> */

/* static */
const prog_EventHandlers UiPage::event_handlers_ PROGMEM = {
  OnInit,
  OnQuit,
  OnIncrement,
  OnClick,
  OnSwitch,
  OnIdle,
  UpdateScreen,
  UpdateLeds,
};

/* static */
void UiPage::OnInit(PageInfo* pageInfo, UiPageIndex prevPage) {
}

/* static */
void UiPage::OnQuit(UiPageIndex nextPage) {
}

/* static */
uint8_t UiPage::OnIncrement(uint8_t id, int8_t value) {
  return 0;
}

/* static */
uint8_t UiPage::OnClick(uint8_t id, uint8_t value) {
  return 0;
}

/* static */
uint8_t UiPage::OnSwitch(uint8_t id, uint8_t value) {
  return 0;
}

/* static */
uint8_t UiPage::OnIdle() {
  return 0;
}

/* static */
uint8_t UiPage::UpdateScreen() {
  return 0;
}

/* static */
uint8_t UiPage::UpdateLeds() {
  return 0;
}

///////////////////////////////////////////////////////////////////////////////
// Miscellaneous helpers

/* static */
uint8_t UiPage::cell_pos(uint8_t step)
{
  // 0000000000111111111122222222223333333333
  // 0123456789012345678901234567890123456789
  // ----|----|----|----||----|----|----|----

  static const prog_uint8_t step_cell_pos[8] PROGMEM = {
    0, 5, 10, 15, 21, 26, 31, 36,
  };

  return pgm_read_byte(&step_cell_pos[step]);
}
  
/* static */
void UiPage::DrawSeparators() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);
  for (uint8_t n = 4; n < kLcdWidth; ) {
    line1[n] = '|'; 
    line2[n] = '|'; 
    if (n == 19) {
      n++;
    } else
      n+= 5;
  }
}

/* static */
void UiPage::DrawSeparators(char* line) {
  for (uint8_t n = 4; n < kLcdWidth; ) {
    line[n] = '|'; 
    if (n == 19) {
      n++;
    } else
      n+= 5;
  }
}

/* static */
void UiPage::DrawStepIcons(char* line) {
  for (uint8_t n = 0; n < kNumSteps; n++) {
    uint8_t hasCond = dev.view_seq_step_has_condition(n);
    uint8_t hasTrig = dev.view_seq_step_has_retrigger(n);
    uint8_t ch;
    if (hasCond && hasTrig) ch = kCondAndTrigChar; else 
    if (hasCond) ch = kCondChar; else 
    if (hasTrig) ch = kTrigChar; else
      continue;
    line[cell_pos(n)] = ch;
  }
}

/* static */
void UiPage::DrawCells(char* line, const prog_char* cells) {
  for (uint8_t n = 0; n < kNumSteps; n++) {
    uint8_t x = cell_pos(n);
    memcpy_P(&line[x], &cells[n << 2], 4);
  }
}

/* static */
void UiPage::DrawSelStr2(char* buffer, uint8_t sel, const prog_char* strings) {
  memcpy_P(buffer, &strings[sel << 1], 2);
}

/* static */
void UiPage::DrawSelStr4(char* buffer, uint8_t sel, const prog_char* strings) {
  memcpy_P(buffer, &strings[sel << 2], 4);
}

/* static */
void UiPage::DrawSelStrN(char* buffer, uint8_t sel, const prog_char* strings, uint8_t width) {
  memcpy_P(buffer, &strings[sel * width], width);
}

/* static */
void UiPage::DrawBrackets(char* buffer, uint8_t max_width) {
  *buffer++ = '[';
  for (uint8_t n = 0; n < max_width; n++, buffer++) {
    if (buffer[1] == ' ' || buffer[1] == '|') {
      buffer[1] = ']';
      break;
    }
  }
}

/* static */
void UiPage::DrawDashes(char* buffer, uint8_t count) {
  for (uint8_t n = 0; n < count; n++, buffer++) {
    buffer[0] = '-';
  }
}

/* static */
uint8_t UiPage::FindLastSpace(char* str, uint8_t count) {
  while (count > 0 && str[count - 1] == ' ') {
    count--;
  }
  return count;
}

}  // namespace midialf
