// Copyright 2012 Peter Kvitek.
//
// Author: Peter Kvitek (pete@kvitek.com)
//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
// -----------------------------------------------------------------------------
//
// UI page for velocity/skip editing

#include "midialf/ui_pages/velocity_page.h"

namespace midialf {

/* static */
const prog_EventHandlers VelocityPage::event_handlers_ PROGMEM = {
  OnInit,
  OnQuit,
  OnIncrement,
  OnClick,
  OnSwitch,
  OnIdle,
  UpdateScreen,
  UpdateLeds,
};

/* static */
void VelocityPage::OnInit(PageInfo* pageInfo, UiPageIndex prevPage) {
}

/* static */
void VelocityPage::OnQuit(UiPageIndex nextPage) {
}

/* static */
uint8_t VelocityPage::OnIncrement(uint8_t id, int8_t value) {
  // ENCA changes all step notes velocity up/down
  if (id == ENCODER_A) {
    for (uint8_t n = 0; n < kNumSteps; n++) {
      uint8_t velo = dev.view_seq_step_velocity(n);
      dev.set_view_seq_step_velocity(n, Clamp(static_cast<int16_t>(velo) + value, 1, 0x7f));
    }
    return 1;
  }
  // ENCx changes step note velocity up/down
  if (id >= ENCODER_1 && id <= ENCODER_8) {
    uint8_t step = id - ENCODER_1;
    uint8_t velo = dev.view_seq_step_velocity(step);
    dev.set_view_seq_step_velocity(step, Clamp7F(static_cast<int16_t>(velo) + value));
    dev.SendViewSeqStep(step);
    return 1;
  }
  return 0;
}

/* static */
uint8_t VelocityPage::OnClick(uint8_t id, uint8_t value) {
  return 0;
}

/* static */
uint8_t VelocityPage::OnSwitch(uint8_t id, uint8_t value) {
  // Step switch toggles step skip
  if (id == SWITCH && value < kNumSteps) {
    uint8_t skip = dev.view_seq_step_skip(value);
    dev.set_view_seq_step_skip(value, !skip);
    return 1;
  }
  return 0;
}

/* static */
uint8_t VelocityPage::OnIdle() {
  return 0;
}

/* static */
uint8_t VelocityPage::UpdateScreen() {
  char* line1 = display.line_buffer(0);
  char* line2 = display.line_buffer(1);

  DrawSeparators();
  DrawStepIcons(line1);

  for (uint8_t n = 0; n < kNumSteps; n++) {
    uint8_t x = cell_pos(n);
    UnsafeItoa(dev.view_seq_step_velocity(n), 3, &line1[x + 1]);
    if (dev.view_seq_step_skip(n)) {
      memcpy_P(&line2[x], PSTRN("skip"));
    }
  }

  return 1;
}

/* static */
uint8_t VelocityPage::UpdateLeds() {
  return 0;
}

} // namespace midialf
